<?php

namespace App\Http\Controllers\Api;

use App\Models\ChartOfAccount;
use App\Models\JournalEntry;
use App\Models\JournalEntryLine;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ChartOfAccountController extends ApiController
{
    public function index(Request $request)
    {
        $query = ChartOfAccount::where('tenant_id', $request->user()->tenant_id);

        if ($request->type) {
            $query->where('type', $request->type);
        }

        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', "%{$request->search}%")
                  ->orWhere('code', 'like', "%{$request->search}%");
            });
        }

        // Return as tree structure
        if ($request->boolean('tree')) {
            $accounts = $query->whereNull('parent_id')
                ->with('children')
                ->orderBy('code')
                ->get();
        } else {
            $accounts = $query->orderBy('code')->get();
        }

        return $this->success($accounts);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'code' => 'required|string|max:20|unique:chart_of_accounts,code,NULL,id,tenant_id,' . $request->user()->tenant_id,
            'name' => 'required|string|max:255',
            'type' => 'required|in:asset,liability,equity,revenue,expense',
            'parent_id' => 'nullable|exists:chart_of_accounts,id',
            'description' => 'nullable|string|max:500',
            'is_active' => 'boolean',
        ]);

        $account = ChartOfAccount::create([
            'tenant_id' => $request->user()->tenant_id,
            'normal_balance' => in_array($validated['type'], ['asset', 'expense']) ? 'debit' : 'credit',
            ...$validated,
        ]);

        return $this->success($account, 'Account created', 201);
    }

    public function show(ChartOfAccount $account)
    {
        $this->authorize('view', $account);

        return $this->success($account->load('parent', 'children'));
    }

    public function update(Request $request, ChartOfAccount $account)
    {
        $this->authorize('update', $account);

        $validated = $request->validate([
            'code' => 'sometimes|string|max:20|unique:chart_of_accounts,code,' . $account->id . ',id,tenant_id,' . $request->user()->tenant_id,
            'name' => 'sometimes|string|max:255',
            'parent_id' => 'nullable|exists:chart_of_accounts,id',
            'description' => 'nullable|string|max:500',
            'is_active' => 'boolean',
        ]);

        // Prevent circular reference
        if (isset($validated['parent_id']) && $validated['parent_id'] == $account->id) {
            return $this->error('Account cannot be its own parent', 400);
        }

        $account->update($validated);

        return $this->success($account, 'Account updated');
    }

    public function destroy(ChartOfAccount $account)
    {
        $this->authorize('delete', $account);

        // Check if account has transactions
        $hasTransactions = JournalEntryLine::where('account_id', $account->id)->exists();
        
        if ($hasTransactions) {
            return $this->error('Cannot delete account with transactions', 400);
        }

        // Check if account has children
        if ($account->children()->exists()) {
            return $this->error('Cannot delete account with sub-accounts', 400);
        }

        $account->delete();

        return $this->success(null, 'Account deleted');
    }

    public function ledger(ChartOfAccount $account, Request $request)
    {
        $this->authorize('view', $account);

        $query = JournalEntryLine::where('account_id', $account->id)
            ->with(['journalEntry' => fn($q) => $q->select('id', 'entry_number', 'entry_date', 'description', 'status')])
            ->whereHas('journalEntry', fn($q) => $q->where('status', 'posted'));

        if ($request->start_date) {
            $query->whereHas('journalEntry', fn($q) => $q->where('entry_date', '>=', $request->start_date));
        }

        if ($request->end_date) {
            $query->whereHas('journalEntry', fn($q) => $q->where('entry_date', '<=', $request->end_date));
        }

        $lines = $query->orderBy('created_at')->get();

        // Calculate running balance
        $runningBalance = 0;
        $ledger = $lines->map(function ($line) use (&$runningBalance, $account) {
            if ($account->normal_balance === 'debit') {
                $runningBalance += $line->debit - $line->credit;
            } else {
                $runningBalance += $line->credit - $line->debit;
            }

            return [
                'date' => $line->journalEntry->entry_date,
                'entry_number' => $line->journalEntry->entry_number,
                'description' => $line->description ?? $line->journalEntry->description,
                'debit' => $line->debit,
                'credit' => $line->credit,
                'balance' => $runningBalance,
            ];
        });

        return $this->success([
            'account' => $account,
            'opening_balance' => 0, // TODO: Calculate from previous period
            'entries' => $ledger,
            'closing_balance' => $runningBalance,
        ]);
    }

    public function trialBalance(Request $request)
    {
        $tenantId = $request->user()->tenant_id;
        $asOfDate = $request->as_of_date ?? now()->format('Y-m-d');

        $accounts = ChartOfAccount::where('tenant_id', $tenantId)
            ->where('is_active', true)
            ->orderBy('code')
            ->get()
            ->map(function ($account) use ($asOfDate) {
                $totals = JournalEntryLine::where('account_id', $account->id)
                    ->whereHas('journalEntry', fn($q) => 
                        $q->where('status', 'posted')
                          ->where('entry_date', '<=', $asOfDate)
                    )
                    ->selectRaw('SUM(debit) as total_debit, SUM(credit) as total_credit')
                    ->first();

                $debit = $totals->total_debit ?? 0;
                $credit = $totals->total_credit ?? 0;

                // Calculate balance based on normal balance
                if ($account->normal_balance === 'debit') {
                    $balance = $debit - $credit;
                    return [
                        'code' => $account->code,
                        'name' => $account->name,
                        'type' => $account->type,
                        'debit' => $balance > 0 ? $balance : 0,
                        'credit' => $balance < 0 ? abs($balance) : 0,
                    ];
                } else {
                    $balance = $credit - $debit;
                    return [
                        'code' => $account->code,
                        'name' => $account->name,
                        'type' => $account->type,
                        'debit' => $balance < 0 ? abs($balance) : 0,
                        'credit' => $balance > 0 ? $balance : 0,
                    ];
                }
            })
            ->filter(fn($a) => $a['debit'] > 0 || $a['credit'] > 0);

        $totalDebit = $accounts->sum('debit');
        $totalCredit = $accounts->sum('credit');

        return $this->success([
            'as_of_date' => $asOfDate,
            'accounts' => $accounts->values(),
            'totals' => [
                'debit' => $totalDebit,
                'credit' => $totalCredit,
                'is_balanced' => abs($totalDebit - $totalCredit) < 0.01,
            ],
        ]);
    }

    public function balanceSheet(Request $request)
    {
        $tenantId = $request->user()->tenant_id;
        $asOfDate = $request->as_of_date ?? now()->format('Y-m-d');

        $getBalanceByType = function ($type) use ($tenantId, $asOfDate) {
            return ChartOfAccount::where('tenant_id', $tenantId)
                ->where('type', $type)
                ->where('is_active', true)
                ->orderBy('code')
                ->get()
                ->map(function ($account) use ($asOfDate) {
                    $totals = JournalEntryLine::where('account_id', $account->id)
                        ->whereHas('journalEntry', fn($q) => 
                            $q->where('status', 'posted')
                              ->where('entry_date', '<=', $asOfDate)
                        )
                        ->selectRaw('SUM(debit) as total_debit, SUM(credit) as total_credit')
                        ->first();

                    $balance = $account->normal_balance === 'debit'
                        ? ($totals->total_debit ?? 0) - ($totals->total_credit ?? 0)
                        : ($totals->total_credit ?? 0) - ($totals->total_debit ?? 0);

                    return [
                        'code' => $account->code,
                        'name' => $account->name,
                        'balance' => $balance,
                    ];
                })
                ->filter(fn($a) => abs($a['balance']) > 0.001);
        };

        $assets = $getBalanceByType('asset');
        $liabilities = $getBalanceByType('liability');
        $equity = $getBalanceByType('equity');

        $totalAssets = $assets->sum('balance');
        $totalLiabilities = $liabilities->sum('balance');
        $totalEquity = $equity->sum('balance');

        return $this->success([
            'as_of_date' => $asOfDate,
            'assets' => [
                'accounts' => $assets->values(),
                'total' => $totalAssets,
            ],
            'liabilities' => [
                'accounts' => $liabilities->values(),
                'total' => $totalLiabilities,
            ],
            'equity' => [
                'accounts' => $equity->values(),
                'total' => $totalEquity,
            ],
            'is_balanced' => abs($totalAssets - $totalLiabilities - $totalEquity) < 0.01,
        ]);
    }

    public function incomeStatement(Request $request)
    {
        $tenantId = $request->user()->tenant_id;
        $startDate = $request->start_date ?? now()->startOfMonth()->format('Y-m-d');
        $endDate = $request->end_date ?? now()->format('Y-m-d');

        $getBalanceByType = function ($type) use ($tenantId, $startDate, $endDate) {
            return ChartOfAccount::where('tenant_id', $tenantId)
                ->where('type', $type)
                ->where('is_active', true)
                ->orderBy('code')
                ->get()
                ->map(function ($account) use ($startDate, $endDate) {
                    $totals = JournalEntryLine::where('account_id', $account->id)
                        ->whereHas('journalEntry', fn($q) => 
                            $q->where('status', 'posted')
                              ->whereBetween('entry_date', [$startDate, $endDate])
                        )
                        ->selectRaw('SUM(debit) as total_debit, SUM(credit) as total_credit')
                        ->first();

                    $balance = $account->normal_balance === 'debit'
                        ? ($totals->total_debit ?? 0) - ($totals->total_credit ?? 0)
                        : ($totals->total_credit ?? 0) - ($totals->total_debit ?? 0);

                    return [
                        'code' => $account->code,
                        'name' => $account->name,
                        'balance' => $balance,
                    ];
                })
                ->filter(fn($a) => abs($a['balance']) > 0.001);
        };

        $revenue = $getBalanceByType('revenue');
        $expenses = $getBalanceByType('expense');

        $totalRevenue = $revenue->sum('balance');
        $totalExpenses = $expenses->sum('balance');
        $netIncome = $totalRevenue - $totalExpenses;

        return $this->success([
            'period' => [
                'start_date' => $startDate,
                'end_date' => $endDate,
            ],
            'revenue' => [
                'accounts' => $revenue->values(),
                'total' => $totalRevenue,
            ],
            'expenses' => [
                'accounts' => $expenses->values(),
                'total' => $totalExpenses,
            ],
            'net_income' => $netIncome,
        ]);
    }
}
