<?php

namespace App\Http\Controllers\Api;

use App\Models\Email;
use App\Models\EmailAccount;
use App\Models\EmailTemplate;
use App\Models\Contact;
use App\Jobs\SendEmail;
use Illuminate\Http\Request;

class EmailController extends ApiController
{
    public function index(Request $request)
    {
        $query = Email::where('tenant_id', $request->user()->tenant_id)
            ->with(['contact', 'emailAccount']);

        if ($request->direction) {
            $query->where('direction', $request->direction);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->contact_id) {
            $query->where('contact_id', $request->contact_id);
        }

        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('subject', 'like', "%{$request->search}%")
                  ->orWhere('body_text', 'like', "%{$request->search}%");
            });
        }

        $emails = $query->latest()->paginate($request->per_page ?? 15);

        return $this->paginated($emails);
    }

    public function show(Email $email)
    {
        $this->authorize('view', $email);

        return $this->success($email->load(['contact', 'emailAccount', 'attachments']));
    }

    public function compose(Request $request)
    {
        $tenantId = $request->user()->tenant_id;

        return $this->success([
            'accounts' => EmailAccount::where('tenant_id', $tenantId)
                ->where('is_active', true)
                ->get(['id', 'name', 'email']),
            'templates' => EmailTemplate::where('tenant_id', $tenantId)
                ->where('is_active', true)
                ->get(['id', 'name', 'subject']),
        ]);
    }

    public function send(Request $request)
    {
        $validated = $request->validate([
            'email_account_id' => 'required|exists:email_accounts,id',
            'to' => 'required|array|min:1',
            'to.*' => 'email',
            'cc' => 'nullable|array',
            'cc.*' => 'email',
            'bcc' => 'nullable|array',
            'bcc.*' => 'email',
            'subject' => 'required|string|max:255',
            'body_html' => 'required|string',
            'body_text' => 'nullable|string',
            'contact_id' => 'nullable|exists:contacts,id',
            'template_id' => 'nullable|exists:email_templates,id',
            'attachments' => 'nullable|array',
            'attachments.*' => 'file|max:10240',
        ]);

        $emailAccount = EmailAccount::findOrFail($validated['email_account_id']);

        if (!$emailAccount->canSend()) {
            return $this->error('Email account has reached daily limit', 429);
        }

        $email = Email::create([
            'tenant_id' => $request->user()->tenant_id,
            'email_account_id' => $validated['email_account_id'],
            'contact_id' => $validated['contact_id'] ?? null,
            'direction' => 'outbound',
            'from_email' => $emailAccount->email,
            'from_name' => $emailAccount->name,
            'to' => $validated['to'],
            'cc' => $validated['cc'] ?? [],
            'bcc' => $validated['bcc'] ?? [],
            'subject' => $validated['subject'],
            'body_html' => $validated['body_html'],
            'body_text' => $validated['body_text'] ?? strip_tags($validated['body_html']),
            'status' => 'queued',
        ]);

        // Handle attachments
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('email-attachments/' . $email->id, 'private');
                $email->attachments()->create([
                    'tenant_id' => $request->user()->tenant_id,
                    'name' => $file->getClientOriginalName(),
                    'path' => $path,
                    'mime_type' => $file->getMimeType(),
                    'size' => $file->getSize(),
                ]);
            }
        }

        // Queue email for sending
        SendEmail::dispatch($email);

        return $this->success($email, 'Email queued for sending', 201);
    }

    public function reply(Request $request, Email $email)
    {
        $this->authorize('view', $email);

        $validated = $request->validate([
            'body_html' => 'required|string',
            'body_text' => 'nullable|string',
            'cc' => 'nullable|array',
            'cc.*' => 'email',
        ]);

        $replyEmail = Email::create([
            'tenant_id' => $request->user()->tenant_id,
            'email_account_id' => $email->email_account_id,
            'contact_id' => $email->contact_id,
            'direction' => 'outbound',
            'from_email' => $email->emailAccount->email,
            'from_name' => $email->emailAccount->name,
            'to' => [$email->from_email],
            'cc' => $validated['cc'] ?? [],
            'subject' => 'Re: ' . $email->subject,
            'body_html' => $validated['body_html'],
            'body_text' => $validated['body_text'] ?? strip_tags($validated['body_html']),
            'in_reply_to' => $email->message_id,
            'thread_id' => $email->thread_id ?? $email->message_id,
            'status' => 'queued',
        ]);

        SendEmail::dispatch($replyEmail);

        return $this->success($replyEmail, 'Reply queued', 201);
    }

    public function forward(Request $request, Email $email)
    {
        $this->authorize('view', $email);

        $validated = $request->validate([
            'to' => 'required|array|min:1',
            'to.*' => 'email',
            'body_html' => 'nullable|string',
        ]);

        $forwardBody = ($validated['body_html'] ?? '') . 
            '<br><br>---------- Forwarded message ---------<br>' .
            'From: ' . $email->from_email . '<br>' .
            'Date: ' . $email->created_at->format('M d, Y H:i') . '<br>' .
            'Subject: ' . $email->subject . '<br><br>' .
            $email->body_html;

        $forwardEmail = Email::create([
            'tenant_id' => $request->user()->tenant_id,
            'email_account_id' => $email->email_account_id,
            'direction' => 'outbound',
            'from_email' => $email->emailAccount->email,
            'from_name' => $email->emailAccount->name,
            'to' => $validated['to'],
            'subject' => 'Fwd: ' . $email->subject,
            'body_html' => $forwardBody,
            'body_text' => strip_tags($forwardBody),
            'status' => 'queued',
        ]);

        SendEmail::dispatch($forwardEmail);

        return $this->success($forwardEmail, 'Email forwarded', 201);
    }

    public function schedule(Request $request)
    {
        $validated = $request->validate([
            'email_account_id' => 'required|exists:email_accounts,id',
            'to' => 'required|array|min:1',
            'to.*' => 'email',
            'subject' => 'required|string|max:255',
            'body_html' => 'required|string',
            'scheduled_at' => 'required|date|after:now',
            'contact_id' => 'nullable|exists:contacts,id',
        ]);

        $email = Email::create([
            'tenant_id' => $request->user()->tenant_id,
            'email_account_id' => $validated['email_account_id'],
            'contact_id' => $validated['contact_id'] ?? null,
            'direction' => 'outbound',
            'from_email' => EmailAccount::find($validated['email_account_id'])->email,
            'to' => $validated['to'],
            'subject' => $validated['subject'],
            'body_html' => $validated['body_html'],
            'body_text' => strip_tags($validated['body_html']),
            'status' => 'draft',
            'scheduled_at' => $validated['scheduled_at'],
        ]);

        return $this->success($email, 'Email scheduled', 201);
    }

    public function cancelScheduled(Email $email)
    {
        $this->authorize('update', $email);

        if ($email->status !== 'draft' || !$email->scheduled_at) {
            return $this->error('Cannot cancel this email', 400);
        }

        $email->delete();

        return $this->success(null, 'Scheduled email cancelled');
    }

    public function markRead(Email $email)
    {
        $this->authorize('update', $email);

        $email->update(['read_at' => now()]);

        return $this->success($email);
    }

    public function markUnread(Email $email)
    {
        $this->authorize('update', $email);

        $email->update(['read_at' => null]);

        return $this->success($email);
    }

    public function trash(Email $email)
    {
        $this->authorize('delete', $email);

        $email->delete();

        return $this->success(null, 'Email moved to trash');
    }

    public function threads(Request $request)
    {
        $threads = Email::where('tenant_id', $request->user()->tenant_id)
            ->whereNotNull('thread_id')
            ->select('thread_id')
            ->selectRaw('MAX(created_at) as last_message_at')
            ->selectRaw('COUNT(*) as message_count')
            ->groupBy('thread_id')
            ->orderByDesc('last_message_at')
            ->paginate($request->per_page ?? 15);

        return $this->paginated($threads);
    }

    public function thread(string $threadId, Request $request)
    {
        $emails = Email::where('tenant_id', $request->user()->tenant_id)
            ->where('thread_id', $threadId)
            ->with(['contact', 'attachments'])
            ->orderBy('created_at')
            ->get();

        return $this->success($emails);
    }
}
