<?php

namespace App\Http\Controllers\Api;

use App\Models\Employee;
use App\Models\EmployeeDocument;
use App\Models\EmployeeSalary;
use App\Services\LeaveService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Storage;

class EmployeeController extends ApiController
{
    public function __construct(protected LeaveService $leaveService) {}

    public function index(Request $request): JsonResponse
    {
        $query = Employee::where('tenant_id', $request->user()->tenant_id)
            ->with(['user:id,first_name,last_name,email', 'department:id,name']);

        if ($request->search) {
            $query->whereHas('user', fn($q) => 
                $q->where('first_name', 'like', "%{$request->search}%")
                  ->orWhere('last_name', 'like', "%{$request->search}%")
            )->orWhere('employee_id', 'like', "%{$request->search}%");
        }

        if ($request->department_id) {
            $query->where('department_id', $request->department_id);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        $employees = $query->orderBy('created_at', 'desc')
            ->paginate($request->per_page ?? 50);

        return $this->paginated($employees);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id|unique:employees,user_id',
            'employee_id' => 'nullable|string|max:50|unique:employees,employee_id',
            'department_id' => 'nullable|exists:departments,id',
            'manager_id' => 'nullable|exists:employees,id',
            'hire_date' => 'required|date',
            'job_title' => 'required|string|max:255',
            'employment_type' => 'required|in:full_time,part_time,contract,intern',
            'work_location' => 'nullable|string|max:255',
            'nationality' => 'nullable|string|size:2',
            'date_of_birth' => 'nullable|date|before:today',
            'gender' => 'nullable|in:male,female,other',
            'marital_status' => 'nullable|in:single,married,divorced,widowed',
            'cpr_number' => 'nullable|string|max:20',
            'passport_number' => 'nullable|string|max:20',
            'basic_salary' => 'nullable|numeric|min:0',
            'housing_allowance' => 'nullable|numeric|min:0',
            'transport_allowance' => 'nullable|numeric|min:0',
        ]);

        $employee = Employee::create([
            'tenant_id' => $request->user()->tenant_id,
            ...$validated,
            'employee_id' => $validated['employee_id'] ?? $this->generateEmployeeId($request->user()->tenant_id),
            'status' => 'active',
        ]);

        // Create initial salary record
        if (isset($validated['basic_salary'])) {
            EmployeeSalary::create([
                'tenant_id' => $request->user()->tenant_id,
                'employee_id' => $employee->id,
                'effective_date' => $validated['hire_date'],
                'basic_salary' => $validated['basic_salary'],
                'housing_allowance' => $validated['housing_allowance'] ?? 0,
                'transport_allowance' => $validated['transport_allowance'] ?? 0,
                'currency' => 'BHD',
            ]);
        }

        // Initialize leave balances
        $this->leaveService->initializeBalances($employee);

        return $this->success($employee->load(['user', 'department']), 'Employee created', 201);
    }

    public function show(Employee $employee): JsonResponse
    {
        $this->authorize('view', $employee);

        return $this->success($employee->load([
            'user', 'department', 'manager.user', 
            'currentSalary', 'leaveBalances.leaveType'
        ]));
    }

    public function update(Request $request, Employee $employee): JsonResponse
    {
        $this->authorize('update', $employee);

        $validated = $request->validate([
            'department_id' => 'nullable|exists:departments,id',
            'manager_id' => 'nullable|exists:employees,id',
            'job_title' => 'sometimes|string|max:255',
            'employment_type' => 'sometimes|in:full_time,part_time,contract,intern',
            'work_location' => 'nullable|string|max:255',
            'nationality' => 'nullable|string|size:2',
            'date_of_birth' => 'nullable|date',
            'gender' => 'nullable|in:male,female,other',
            'marital_status' => 'nullable|in:single,married,divorced,widowed',
            'cpr_number' => 'nullable|string|max:20',
            'passport_number' => 'nullable|string|max:20',
            'status' => 'sometimes|in:active,on_leave,probation,terminated',
            'termination_date' => 'nullable|date',
            'termination_reason' => 'nullable|string',
        ]);

        $employee->update($validated);
        return $this->success($employee->load(['user', 'department']), 'Employee updated');
    }

    public function destroy(Employee $employee): JsonResponse
    {
        $this->authorize('delete', $employee);
        $employee->delete();
        return $this->success(null, 'Employee deleted');
    }

    public function documents(Employee $employee): JsonResponse
    {
        $this->authorize('view', $employee);

        $documents = $employee->documents()->orderBy('created_at', 'desc')->get();
        return $this->success($documents);
    }

    public function uploadDocument(Request $request, Employee $employee): JsonResponse
    {
        $this->authorize('update', $employee);

        $validated = $request->validate([
            'document' => 'required|file|max:10240',
            'type' => 'required|in:contract,id,passport,visa,certificate,other',
            'name' => 'required|string|max:255',
            'expiry_date' => 'nullable|date',
        ]);

        $path = $request->file('document')->store("employees/{$employee->id}/documents", 'private');

        $document = EmployeeDocument::create([
            'tenant_id' => $employee->tenant_id,
            'employee_id' => $employee->id,
            'type' => $validated['type'],
            'name' => $validated['name'],
            'file_path' => $path,
            'file_size' => $request->file('document')->getSize(),
            'mime_type' => $request->file('document')->getMimeType(),
            'expiry_date' => $validated['expiry_date'] ?? null,
            'uploaded_by' => $request->user()->id,
        ]);

        return $this->success($document, 'Document uploaded', 201);
    }

    public function salaryHistory(Employee $employee): JsonResponse
    {
        $this->authorize('view', $employee);

        $history = $employee->salaryHistory()->orderBy('effective_date', 'desc')->get();
        return $this->success($history);
    }

    public function updateSalary(Request $request, Employee $employee): JsonResponse
    {
        $this->authorize('update', $employee);

        $validated = $request->validate([
            'effective_date' => 'required|date',
            'basic_salary' => 'required|numeric|min:0',
            'housing_allowance' => 'nullable|numeric|min:0',
            'transport_allowance' => 'nullable|numeric|min:0',
            'other_allowances' => 'nullable|numeric|min:0',
            'reason' => 'nullable|string|max:255',
        ]);

        $salary = EmployeeSalary::create([
            'tenant_id' => $employee->tenant_id,
            'employee_id' => $employee->id,
            ...$validated,
            'currency' => 'BHD',
        ]);

        return $this->success($salary, 'Salary updated', 201);
    }

    protected function generateEmployeeId(int $tenantId): string
    {
        $count = Employee::where('tenant_id', $tenantId)->count() + 1;
        return 'EMP-' . str_pad($count, 4, '0', STR_PAD_LEFT);
    }
}
