<?php

namespace App\Http\Controllers\Api;

use App\Models\Expense;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ExpenseController extends ApiController
{
    public function index(Request $request)
    {
        $query = Expense::where('tenant_id', $request->user()->tenant_id)
            ->with(['account', 'vendor', 'contact', 'createdBy']);

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->category) {
            $query->where('category', $request->category);
        }

        if ($request->vendor_id) {
            $query->where('vendor_id', $request->vendor_id);
        }

        if ($request->from_date) {
            $query->whereDate('expense_date', '>=', $request->from_date);
        }

        if ($request->to_date) {
            $query->whereDate('expense_date', '<=', $request->to_date);
        }

        if ($request->has('is_billable')) {
            $query->where('is_billable', $request->boolean('is_billable'));
        }

        if ($request->has('is_reimbursable')) {
            $query->where('is_reimbursable', $request->boolean('is_reimbursable'));
        }

        $expenses = $query->orderBy('expense_date', 'desc')
            ->paginate($request->per_page ?? 15);

        return $this->paginated($expenses);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'account_id' => 'required|exists:chart_of_accounts,id',
            'vendor_id' => 'nullable|exists:vendors,id',
            'contact_id' => 'nullable|exists:contacts,id',
            'expense_date' => 'required|date',
            'amount' => 'required|numeric|min:0.01',
            'currency' => 'nullable|string|size:3',
            'tax_rate_id' => 'nullable|exists:tax_rates,id',
            'category' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'receipt' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:5120',
            'is_billable' => 'boolean',
            'is_reimbursable' => 'boolean',
            'payment_method' => 'nullable|in:cash,bank_transfer,credit_card,cheque',
            'reference' => 'nullable|string|max:100',
        ]);

        $expense = new Expense([
            'tenant_id' => $request->user()->tenant_id,
            'account_id' => $validated['account_id'],
            'vendor_id' => $validated['vendor_id'] ?? null,
            'contact_id' => $validated['contact_id'] ?? null,
            'expense_date' => $validated['expense_date'],
            'amount' => $validated['amount'],
            'currency' => $validated['currency'] ?? 'BHD',
            'tax_rate_id' => $validated['tax_rate_id'] ?? null,
            'category' => $validated['category'] ?? null,
            'description' => $validated['description'] ?? null,
            'is_billable' => $validated['is_billable'] ?? false,
            'is_reimbursable' => $validated['is_reimbursable'] ?? false,
            'payment_method' => $validated['payment_method'] ?? null,
            'reference' => $validated['reference'] ?? null,
            'created_by' => $request->user()->id,
        ]);

        $expense->calculateTotal();

        if ($request->hasFile('receipt')) {
            $path = $request->file('receipt')->store('expenses/receipts', 'public');
            $expense->receipt_path = $path;
        }

        $expense->save();

        return $this->success($expense->load('account', 'vendor'), 'Expense created successfully', 201);
    }

    public function show(Expense $expense)
    {
        return $this->success($expense->load('account', 'vendor', 'contact', 'createdBy', 'approvedBy'));
    }

    public function update(Request $request, Expense $expense)
    {
        if ($expense->status !== Expense::STATUS_PENDING) {
            return $this->error('Only pending expenses can be edited', 400);
        }

        $validated = $request->validate([
            'account_id' => 'sometimes|exists:chart_of_accounts,id',
            'vendor_id' => 'nullable|exists:vendors,id',
            'expense_date' => 'sometimes|date',
            'amount' => 'sometimes|numeric|min:0.01',
            'tax_rate_id' => 'nullable|exists:tax_rates,id',
            'category' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'is_billable' => 'boolean',
            'is_reimbursable' => 'boolean',
        ]);

        $expense->update($validated);
        $expense->calculateTotal();
        $expense->save();

        return $this->success($expense, 'Expense updated successfully');
    }

    public function destroy(Expense $expense)
    {
        if ($expense->status !== Expense::STATUS_PENDING) {
            return $this->error('Only pending expenses can be deleted', 400);
        }

        if ($expense->receipt_path) {
            Storage::disk('public')->delete($expense->receipt_path);
        }

        $expense->delete();

        return $this->success(null, 'Expense deleted successfully');
    }

    public function approve(Expense $expense)
    {
        if ($expense->status !== Expense::STATUS_PENDING) {
            return $this->error('Expense is not pending approval', 400);
        }

        $expense->approve(auth()->user());

        return $this->success($expense, 'Expense approved successfully');
    }

    public function reject(Request $request, Expense $expense)
    {
        if ($expense->status !== Expense::STATUS_PENDING) {
            return $this->error('Expense is not pending approval', 400);
        }

        $validated = $request->validate([
            'reason' => 'required|string|max:500',
        ]);

        $expense->reject(auth()->user(), $validated['reason']);

        return $this->success($expense, 'Expense rejected');
    }

    public function uploadReceipt(Request $request, Expense $expense)
    {
        $request->validate([
            'receipt' => 'required|file|mimes:jpg,jpeg,png,pdf|max:5120',
        ]);

        if ($expense->receipt_path) {
            Storage::disk('public')->delete($expense->receipt_path);
        }

        $path = $request->file('receipt')->store('expenses/receipts', 'public');
        $expense->update(['receipt_path' => $path]);

        return $this->success($expense, 'Receipt uploaded successfully');
    }

    public function byCategory(Request $request)
    {
        $tenantId = $request->user()->tenant_id;
        $fromDate = $request->from_date ?? now()->startOfMonth();
        $toDate = $request->to_date ?? now();

        $expenses = Expense::where('tenant_id', $tenantId)
            ->where('status', Expense::STATUS_APPROVED)
            ->whereBetween('expense_date', [$fromDate, $toDate])
            ->selectRaw('category, SUM(total) as total_amount, COUNT(*) as count')
            ->groupBy('category')
            ->orderBy('total_amount', 'desc')
            ->get();

        return $this->success($expenses);
    }

    public function summary(Request $request)
    {
        $tenantId = $request->user()->tenant_id;
        $fromDate = $request->from_date ?? now()->startOfMonth();
        $toDate = $request->to_date ?? now();

        $query = Expense::where('tenant_id', $tenantId)
            ->whereBetween('expense_date', [$fromDate, $toDate]);

        return $this->success([
            'total' => (clone $query)->sum('total'),
            'pending' => (clone $query)->where('status', 'pending')->sum('total'),
            'approved' => (clone $query)->where('status', 'approved')->sum('total'),
            'billable' => (clone $query)->where('is_billable', true)->sum('total'),
            'reimbursable' => (clone $query)->where('is_reimbursable', true)->sum('total'),
            'count' => (clone $query)->count(),
        ]);
    }
}
