<?php

namespace App\Http\Controllers\Api;

use App\Models\JournalEntry;
use App\Models\JournalEntryLine;
use App\Models\ChartOfAccount;
use App\Models\FiscalYear;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class JournalEntryController extends ApiController
{
    public function index(Request $request)
    {
        $query = JournalEntry::where('tenant_id', $request->user()->tenant_id)
            ->with(['lines.account', 'createdBy']);

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->start_date) {
            $query->where('entry_date', '>=', $request->start_date);
        }

        if ($request->end_date) {
            $query->where('entry_date', '<=', $request->end_date);
        }

        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('entry_number', 'like', "%{$request->search}%")
                  ->orWhere('description', 'like', "%{$request->search}%");
            });
        }

        $entries = $query->latest('entry_date')
            ->paginate($request->per_page ?? 15);

        return $this->paginated($entries);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'entry_date' => 'required|date',
            'description' => 'required|string|max:500',
            'reference' => 'nullable|string|max:100',
            'lines' => 'required|array|min:2',
            'lines.*.account_id' => 'required|exists:chart_of_accounts,id',
            'lines.*.debit' => 'required_without:lines.*.credit|numeric|min:0',
            'lines.*.credit' => 'required_without:lines.*.debit|numeric|min:0',
            'lines.*.description' => 'nullable|string|max:255',
            'lines.*.contact_id' => 'nullable|exists:contacts,id',
        ]);

        // Validate fiscal year is open
        $fiscalYear = FiscalYear::forDate($request->user()->tenant_id, $validated['entry_date']);
        if ($fiscalYear && $fiscalYear->isClosed()) {
            return $this->error('Cannot create entry in closed fiscal year', 400);
        }

        // Validate debits = credits
        $totalDebit = collect($validated['lines'])->sum('debit');
        $totalCredit = collect($validated['lines'])->sum('credit');

        if (abs($totalDebit - $totalCredit) > 0.001) {
            return $this->error('Debits must equal credits', 400);
        }

        return DB::transaction(function () use ($request, $validated) {
            $entry = JournalEntry::create([
                'tenant_id' => $request->user()->tenant_id,
                'entry_date' => $validated['entry_date'],
                'description' => $validated['description'],
                'reference' => $validated['reference'] ?? null,
                'created_by' => auth()->id(),
            ]);

            foreach ($validated['lines'] as $line) {
                $entry->lines()->create([
                    'tenant_id' => $request->user()->tenant_id,
                    'account_id' => $line['account_id'],
                    'debit' => $line['debit'] ?? 0,
                    'credit' => $line['credit'] ?? 0,
                    'description' => $line['description'] ?? null,
                    'contact_id' => $line['contact_id'] ?? null,
                ]);
            }

            return $this->success($entry->load('lines.account'), 'Journal entry created', 201);
        });
    }

    public function show(JournalEntry $entry)
    {
        $this->authorize('view', $entry);

        return $this->success($entry->load(['lines.account', 'lines.contact', 'createdBy']));
    }

    public function update(Request $request, JournalEntry $entry)
    {
        $this->authorize('update', $entry);

        if ($entry->status === JournalEntry::STATUS_POSTED) {
            return $this->error('Cannot edit posted journal entry', 400);
        }

        $validated = $request->validate([
            'entry_date' => 'sometimes|date',
            'description' => 'sometimes|string|max:500',
            'reference' => 'nullable|string|max:100',
            'lines' => 'sometimes|array|min:2',
            'lines.*.account_id' => 'required|exists:chart_of_accounts,id',
            'lines.*.debit' => 'required_without:lines.*.credit|numeric|min:0',
            'lines.*.credit' => 'required_without:lines.*.debit|numeric|min:0',
            'lines.*.description' => 'nullable|string|max:255',
        ]);

        // Validate debits = credits if lines are updated
        if (isset($validated['lines'])) {
            $totalDebit = collect($validated['lines'])->sum('debit');
            $totalCredit = collect($validated['lines'])->sum('credit');

            if (abs($totalDebit - $totalCredit) > 0.001) {
                return $this->error('Debits must equal credits', 400);
            }
        }

        return DB::transaction(function () use ($entry, $validated) {
            $entry->update(collect($validated)->except('lines')->toArray());

            if (isset($validated['lines'])) {
                $entry->lines()->delete();

                foreach ($validated['lines'] as $line) {
                    $entry->lines()->create([
                        'tenant_id' => $entry->tenant_id,
                        'account_id' => $line['account_id'],
                        'debit' => $line['debit'] ?? 0,
                        'credit' => $line['credit'] ?? 0,
                        'description' => $line['description'] ?? null,
                    ]);
                }
            }

            return $this->success($entry->fresh(['lines.account']), 'Journal entry updated');
        });
    }

    public function destroy(JournalEntry $entry)
    {
        $this->authorize('delete', $entry);

        if ($entry->status === JournalEntry::STATUS_POSTED) {
            return $this->error('Cannot delete posted journal entry', 400);
        }

        $entry->delete();

        return $this->success(null, 'Journal entry deleted');
    }

    public function post(JournalEntry $entry)
    {
        $this->authorize('update', $entry);

        if ($entry->status === JournalEntry::STATUS_POSTED) {
            return $this->error('Entry already posted', 400);
        }

        if (!$entry->isBalanced()) {
            return $this->error('Entry is not balanced', 400);
        }

        $entry->post();

        return $this->success($entry, 'Journal entry posted');
    }

    public function unpost(JournalEntry $entry)
    {
        $this->authorize('update', $entry);

        if ($entry->status !== JournalEntry::STATUS_POSTED) {
            return $this->error('Entry is not posted', 400);
        }

        $entry->unpost();

        return $this->success($entry, 'Journal entry unposted');
    }

    public function reverse(JournalEntry $entry)
    {
        $this->authorize('update', $entry);

        if ($entry->status !== JournalEntry::STATUS_POSTED) {
            return $this->error('Can only reverse posted entries', 400);
        }

        $reversal = $entry->reverse();

        return $this->success($reversal->load('lines.account'), 'Reversal entry created', 201);
    }

    public function duplicate(JournalEntry $entry)
    {
        $this->authorize('view', $entry);

        return DB::transaction(function () use ($entry) {
            $newEntry = JournalEntry::create([
                'tenant_id' => $entry->tenant_id,
                'entry_date' => now(),
                'description' => $entry->description . ' (Copy)',
                'reference' => null,
                'status' => JournalEntry::STATUS_DRAFT,
                'created_by' => auth()->id(),
            ]);

            foreach ($entry->lines as $line) {
                $newEntry->lines()->create([
                    'tenant_id' => $entry->tenant_id,
                    'account_id' => $line->account_id,
                    'debit' => $line->debit,
                    'credit' => $line->credit,
                    'description' => $line->description,
                    'contact_id' => $line->contact_id,
                ]);
            }

            return $this->success($newEntry->load('lines.account'), 'Journal entry duplicated', 201);
        });
    }

    public function recurring(Request $request)
    {
        $validated = $request->validate([
            'template_id' => 'required|exists:journal_entries,id',
            'frequency' => 'required|in:monthly,quarterly,yearly',
            'start_date' => 'required|date',
            'end_date' => 'nullable|date|after:start_date',
        ]);

        // Store recurring settings
        // Implementation depends on your recurring system

        return $this->success(null, 'Recurring entry configured');
    }
}
