<?php

namespace App\Http\Controllers\Api;

use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class NotificationController extends ApiController
{
    public function __construct(
        protected NotificationService $notificationService
    ) {}

    public function index(Request $request): JsonResponse
    {
        $notifications = $request->user()->notifications()
            ->orderByDesc('created_at')
            ->paginate($request->get('per_page', 20));

        return $this->paginated($notifications);
    }

    public function unread(Request $request): JsonResponse
    {
        return $this->success(
            $this->notificationService->getUnread($request->user(), $request->get('limit', 20))
        );
    }

    public function unreadCount(Request $request): JsonResponse
    {
        return $this->success([
            'count' => $this->notificationService->getUnreadCount($request->user())
        ]);
    }

    public function markAsRead(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'ids' => 'nullable|array',
            'ids.*' => 'uuid',
        ]);

        $count = $this->notificationService->markAsRead(
            $request->user(),
            $validated['ids'] ?? null
        );

        return $this->success(['marked' => $count], 'Notifications marked as read');
    }

    public function markAllAsRead(Request $request): JsonResponse
    {
        $count = $this->notificationService->markAllAsRead($request->user());
        return $this->success(['marked' => $count], 'All notifications marked as read');
    }

    public function destroy(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'ids' => 'required|array',
            'ids.*' => 'uuid',
        ]);

        $count = $this->notificationService->delete($request->user(), $validated['ids']);
        return $this->success(['deleted' => $count], 'Notifications deleted');
    }

    public function clearAll(Request $request): JsonResponse
    {
        $count = $this->notificationService->clearAll($request->user());
        return $this->success(['deleted' => $count], 'All notifications cleared');
    }

    public function settings(Request $request): JsonResponse
    {
        return $this->success($request->user()->notification_settings ?? []);
    }

    public function updateSettings(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'email' => 'nullable|array',
            'push' => 'nullable|array',
            'sms' => 'nullable|array',
        ]);

        $request->user()->update([
            'notification_settings' => $validated
        ]);

        return $this->success($validated, 'Notification settings updated');
    }
}
