<?php

namespace App\Http\Controllers\Api;

use App\Models\Opportunity;
use App\Models\Pipeline;
use App\Models\Activity;
use Illuminate\Http\Request;

class OpportunityController extends ApiController
{
    public function index(Request $request)
    {
        $query = Opportunity::where('tenant_id', $request->user()->tenant_id)
            ->with(['contact', 'pipeline', 'stage', 'owner']);

        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('title', 'like', "%{$request->search}%")
                  ->orWhereHas('contact', fn($c) => 
                      $c->where('first_name', 'like', "%{$request->search}%")
                        ->orWhere('last_name', 'like', "%{$request->search}%")
                        ->orWhere('email', 'like', "%{$request->search}%")
                  );
            });
        }

        if ($request->pipeline_id) {
            $query->where('pipeline_id', $request->pipeline_id);
        }

        if ($request->stage_id) {
            $query->where('stage_id', $request->stage_id);
        }

        if ($request->owner_id) {
            $query->where('owner_id', $request->owner_id);
        }

        if ($request->status) {
            $query->where('status', $request->status);
        }

        if ($request->min_value) {
            $query->where('value', '>=', $request->min_value);
        }

        if ($request->max_value) {
            $query->where('value', '<=', $request->max_value);
        }

        $opportunities = $query
            ->orderBy($request->sort_by ?? 'created_at', $request->sort_direction ?? 'desc')
            ->paginate($request->per_page ?? 15);

        return $this->paginated($opportunities);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'contact_id' => 'required|exists:contacts,id',
            'pipeline_id' => 'required|exists:pipelines,id',
            'stage_id' => 'nullable|exists:pipeline_stages,id',
            'value' => 'nullable|numeric|min:0',
            'currency' => 'nullable|string|size:3',
            'probability' => 'nullable|integer|min:0|max:100',
            'expected_close_date' => 'nullable|date',
            'owner_id' => 'nullable|exists:users,id',
            'source' => 'nullable|string|max:100',
            'description' => 'nullable|string',
        ]);

        // Get default stage if not provided
        if (empty($validated['stage_id'])) {
            $pipeline = Pipeline::findOrFail($validated['pipeline_id']);
            $validated['stage_id'] = $pipeline->stages()->orderBy('position')->first()?->id;
        }

        $opportunity = Opportunity::create([
            'tenant_id' => $request->user()->tenant_id,
            'owner_id' => $validated['owner_id'] ?? $request->user()->id,
            'currency' => $validated['currency'] ?? 'BHD',
            ...$validated,
        ]);

        // Log activity
        Activity::log($opportunity->contact, 'opportunity_created', [
            'subject' => "Opportunity created: {$opportunity->title}",
            'metadata' => ['opportunity_id' => $opportunity->id],
        ]);

        return $this->success($opportunity->load(['contact', 'pipeline', 'stage', 'owner']), 'Opportunity created', 201);
    }

    public function show(Opportunity $opportunity)
    {
        $this->authorize('view', $opportunity);

        return $this->success($opportunity->load([
            'contact', 'pipeline', 'stage', 'owner',
            'activities' => fn($q) => $q->latest()->limit(10),
            'products',
        ]));
    }

    public function update(Request $request, Opportunity $opportunity)
    {
        $this->authorize('update', $opportunity);

        $validated = $request->validate([
            'title' => 'sometimes|string|max:255',
            'contact_id' => 'sometimes|exists:contacts,id',
            'pipeline_id' => 'sometimes|exists:pipelines,id',
            'stage_id' => 'sometimes|exists:pipeline_stages,id',
            'value' => 'nullable|numeric|min:0',
            'probability' => 'nullable|integer|min:0|max:100',
            'expected_close_date' => 'nullable|date',
            'owner_id' => 'nullable|exists:users,id',
            'source' => 'nullable|string|max:100',
            'description' => 'nullable|string',
        ]);

        $oldStageId = $opportunity->stage_id;
        $opportunity->update($validated);

        // Log stage change
        if (isset($validated['stage_id']) && $oldStageId !== $validated['stage_id']) {
            Activity::log($opportunity->contact, 'opportunity_stage_changed', [
                'subject' => "Opportunity moved to {$opportunity->stage->name}",
                'metadata' => [
                    'opportunity_id' => $opportunity->id,
                    'old_stage_id' => $oldStageId,
                    'new_stage_id' => $opportunity->stage_id,
                ],
            ]);
        }

        return $this->success($opportunity->load(['contact', 'pipeline', 'stage', 'owner']), 'Opportunity updated');
    }

    public function destroy(Opportunity $opportunity)
    {
        $this->authorize('delete', $opportunity);

        $opportunity->delete();

        return $this->success(null, 'Opportunity deleted');
    }

    public function moveStage(Request $request, Opportunity $opportunity)
    {
        $this->authorize('update', $opportunity);

        $validated = $request->validate([
            'stage_id' => 'required|exists:pipeline_stages,id',
        ]);

        $oldStageId = $opportunity->stage_id;
        $opportunity->update(['stage_id' => $validated['stage_id']]);

        // Update probability based on stage
        $newStage = $opportunity->stage;
        if ($newStage->probability !== null) {
            $opportunity->update(['probability' => $newStage->probability]);
        }

        Activity::log($opportunity->contact, 'opportunity_stage_changed', [
            'subject' => "Opportunity moved to {$newStage->name}",
            'metadata' => [
                'opportunity_id' => $opportunity->id,
                'old_stage_id' => $oldStageId,
                'new_stage_id' => $opportunity->stage_id,
            ],
        ]);

        return $this->success($opportunity->load(['contact', 'pipeline', 'stage', 'owner']), 'Opportunity moved');
    }

    public function markWon(Request $request, Opportunity $opportunity)
    {
        $this->authorize('update', $opportunity);

        $validated = $request->validate([
            'won_reason' => 'nullable|string|max:500',
            'actual_value' => 'nullable|numeric|min:0',
        ]);

        $opportunity->update([
            'status' => 'won',
            'won_at' => now(),
            'won_reason' => $validated['won_reason'] ?? null,
            'value' => $validated['actual_value'] ?? $opportunity->value,
            'probability' => 100,
        ]);

        Activity::log($opportunity->contact, 'opportunity_won', [
            'subject' => "Opportunity won: {$opportunity->title}",
            'metadata' => ['opportunity_id' => $opportunity->id, 'value' => $opportunity->value],
        ]);

        // Update contact status
        $opportunity->contact->update(['status' => 'customer']);

        return $this->success($opportunity, 'Opportunity marked as won');
    }

    public function markLost(Request $request, Opportunity $opportunity)
    {
        $this->authorize('update', $opportunity);

        $validated = $request->validate([
            'lost_reason' => 'required|string|max:500',
            'competitor' => 'nullable|string|max:255',
        ]);

        $opportunity->update([
            'status' => 'lost',
            'lost_at' => now(),
            'lost_reason' => $validated['lost_reason'],
            'lost_to_competitor' => $validated['competitor'] ?? null,
            'probability' => 0,
        ]);

        Activity::log($opportunity->contact, 'opportunity_lost', [
            'subject' => "Opportunity lost: {$opportunity->title}",
            'metadata' => ['opportunity_id' => $opportunity->id, 'reason' => $validated['lost_reason']],
        ]);

        return $this->success($opportunity, 'Opportunity marked as lost');
    }

    public function kanban(Request $request, Pipeline $pipeline)
    {
        $stages = $pipeline->stages()
            ->with(['opportunities' => function ($query) use ($request) {
                $query->where('status', 'open')
                    ->with(['contact', 'owner'])
                    ->orderBy('position');
                    
                if ($request->owner_id) {
                    $query->where('owner_id', $request->owner_id);
                }
            }])
            ->orderBy('position')
            ->get();

        return $this->success([
            'pipeline' => $pipeline,
            'stages' => $stages,
            'totals' => [
                'count' => $stages->sum(fn($s) => $s->opportunities->count()),
                'value' => $stages->sum(fn($s) => $s->opportunities->sum('value')),
            ],
        ]);
    }

    public function summary(Request $request)
    {
        $tenantId = $request->user()->tenant_id;

        $openOpportunities = Opportunity::where('tenant_id', $tenantId)
            ->where('status', 'open');

        return $this->success([
            'total_open' => $openOpportunities->count(),
            'total_value' => $openOpportunities->sum('value'),
            'weighted_value' => $openOpportunities->sum(\DB::raw('value * probability / 100')),
            'won_this_month' => Opportunity::where('tenant_id', $tenantId)
                ->where('status', 'won')
                ->whereMonth('won_at', now()->month)
                ->whereYear('won_at', now()->year)
                ->sum('value'),
            'lost_this_month' => Opportunity::where('tenant_id', $tenantId)
                ->where('status', 'lost')
                ->whereMonth('lost_at', now()->month)
                ->whereYear('lost_at', now()->year)
                ->count(),
            'by_stage' => Opportunity::where('tenant_id', $tenantId)
                ->where('status', 'open')
                ->selectRaw('stage_id, COUNT(*) as count, SUM(value) as value')
                ->groupBy('stage_id')
                ->with('stage:id,name')
                ->get(),
        ]);
    }
}
