<?php

namespace App\Http\Controllers\Api;

use App\Models\PayrollRun;
use App\Services\PayrollService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class PayrollController extends ApiController
{
    public function __construct(protected PayrollService $payrollService) {}

    public function index(Request $request): JsonResponse
    {
        $runs = PayrollRun::where('tenant_id', $request->user()->tenant_id)
            ->withCount('items')
            ->orderBy('created_at', 'desc')
            ->paginate($request->per_page ?? 20);

        return $this->paginated($runs);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'pay_period_start' => 'required|date',
            'pay_period_end' => 'required|date|after:pay_period_start',
            'payment_date' => 'required|date|after_or_equal:pay_period_end',
        ]);

        $run = $this->payrollService->createRun(
            $request->user()->tenant_id,
            $validated,
            $request->user()->id
        );

        return $this->success($run, 'Payroll run created', 201);
    }

    public function show(PayrollRun $run): JsonResponse
    {
        $this->authorize('view', $run);

        return $this->success($run->load(['items.employee.user', 'createdBy', 'approvedBy']));
    }

    public function process(PayrollRun $run): JsonResponse
    {
        $this->authorize('update', $run);

        if ($run->status !== PayrollRun::STATUS_DRAFT) {
            return $this->error('Payroll run must be in draft status to process', 422);
        }

        $this->payrollService->process($run);

        return $this->success($run->fresh()->load('items'), 'Payroll processed');
    }

    public function approve(Request $request, PayrollRun $run): JsonResponse
    {
        $this->authorize('approve', $run);

        if ($run->status !== PayrollRun::STATUS_PROCESSED) {
            return $this->error('Payroll run must be processed before approval', 422);
        }

        $this->payrollService->approve($run, $request->user()->id);

        return $this->success($run->fresh(), 'Payroll approved');
    }

    public function markPaid(PayrollRun $run): JsonResponse
    {
        $this->authorize('update', $run);

        if ($run->status !== PayrollRun::STATUS_APPROVED) {
            return $this->error('Payroll run must be approved before marking as paid', 422);
        }

        $this->payrollService->markAsPaid($run);

        return $this->success($run->fresh(), 'Payroll marked as paid');
    }

    public function payslips(PayrollRun $run): JsonResponse
    {
        $this->authorize('view', $run);

        $items = $run->items()->with('employee.user')->get()->map(fn($item) => [
            'id' => $item->id,
            'employee' => $item->employee->user->full_name,
            'employee_id' => $item->employee->employee_id,
            'basic_salary' => $item->basic_salary,
            'allowances' => $item->housing_allowance + $item->transport_allowance + $item->other_allowances,
            'gross_salary' => $item->gross_salary,
            'deductions' => $item->total_deductions,
            'net_salary' => $item->net_salary,
        ]);

        return $this->success($items);
    }

    public function downloadPayslip(PayrollRun $run, int $itemId): JsonResponse
    {
        $this->authorize('view', $run);

        $item = $run->items()->with(['employee.user', 'employee.department'])->findOrFail($itemId);

        // Generate PDF payslip
        // Return download URL

        return $this->success(['url' => '/payslips/' . $item->id . '.pdf']);
    }
}
