<?php

namespace App\Http\Controllers\Api;

use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class ProductController extends ApiController
{
    public function index(Request $request): JsonResponse
    {
        $query = Product::where('tenant_id', $request->user()->tenant_id);

        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        if ($request->filled('search')) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('sku', 'like', '%' . $request->search . '%');
            });
        }

        $products = $query->orderBy('name')
            ->paginate($request->get('per_page', 50));

        return $this->paginated($products);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'sku' => 'nullable|string|max:100',
            'type' => 'nullable|in:product,service',
            'category' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'unit_price' => 'required|numeric|min:0',
            'cost_price' => 'nullable|numeric|min:0',
            'tax_rate_id' => 'nullable|exists:tax_rates,id',
            'income_account_id' => 'nullable|exists:chart_of_accounts,id',
            'expense_account_id' => 'nullable|exists:chart_of_accounts,id',
            'track_inventory' => 'nullable|boolean',
            'quantity_on_hand' => 'nullable|integer|min:0',
            'reorder_point' => 'nullable|integer|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        $product = Product::create([
            'tenant_id' => $request->user()->tenant_id,
            ...$validated,
            'created_by' => $request->user()->id,
        ]);

        return $this->success($product, 'Product created successfully', 201);
    }

    public function show(Request $request, Product $product): JsonResponse
    {
        $this->authorize('view', $product);
        return $this->success($product->load(['taxRate', 'incomeAccount', 'expenseAccount']));
    }

    public function update(Request $request, Product $product): JsonResponse
    {
        $this->authorize('update', $product);

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'sku' => 'nullable|string|max:100',
            'type' => 'nullable|in:product,service',
            'category' => 'nullable|string|max:100',
            'description' => 'nullable|string',
            'unit_price' => 'sometimes|numeric|min:0',
            'cost_price' => 'nullable|numeric|min:0',
            'tax_rate_id' => 'nullable|exists:tax_rates,id',
            'income_account_id' => 'nullable|exists:chart_of_accounts,id',
            'expense_account_id' => 'nullable|exists:chart_of_accounts,id',
            'track_inventory' => 'nullable|boolean',
            'quantity_on_hand' => 'nullable|integer|min:0',
            'reorder_point' => 'nullable|integer|min:0',
            'is_active' => 'nullable|boolean',
        ]);

        $product->update($validated);

        return $this->success($product->fresh(), 'Product updated successfully');
    }

    public function destroy(Request $request, Product $product): JsonResponse
    {
        $this->authorize('delete', $product);
        $product->delete();
        return $this->success(null, 'Product deleted successfully');
    }

    public function categories(Request $request): JsonResponse
    {
        $categories = Product::where('tenant_id', $request->user()->tenant_id)
            ->whereNotNull('category')
            ->distinct()
            ->pluck('category');

        return $this->success($categories);
    }
}
