<?php

namespace App\Http\Controllers\Api;

use App\Models\Team;
use App\Models\User;
use Illuminate\Http\Request;

class TeamController extends ApiController
{
    public function index(Request $request)
    {
        $teams = Team::where('tenant_id', $request->user()->tenant_id)
            ->withCount('members')
            ->orderBy('name')
            ->get();

        return $this->success($teams);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'lead_id' => 'nullable|exists:users,id',
        ]);

        $team = Team::create([
            'tenant_id' => $request->user()->tenant_id,
            ...$validated,
        ]);

        return $this->success($team, 'Team created successfully', 201);
    }

    public function show(Team $team)
    {
        return $this->success($team->load('members', 'lead'));
    }

    public function update(Request $request, Team $team)
    {
        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'description' => 'nullable|string',
            'lead_id' => 'nullable|exists:users,id',
        ]);

        $team->update($validated);

        return $this->success($team, 'Team updated successfully');
    }

    public function destroy(Team $team)
    {
        if ($team->members()->count() > 0) {
            return $this->error('Cannot delete team with members', 400);
        }

        $team->delete();

        return $this->success(null, 'Team deleted successfully');
    }

    public function members(Team $team)
    {
        return $this->success($team->members);
    }

    public function addMember(Request $request, Team $team)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'role' => 'nullable|string|max:50',
        ]);

        $team->members()->syncWithoutDetaching([
            $validated['user_id'] => ['role' => $validated['role'] ?? 'member'],
        ]);

        return $this->success($team->load('members'), 'Member added successfully');
    }

    public function removeMember(Team $team, User $user)
    {
        $team->members()->detach($user->id);

        return $this->success(null, 'Member removed successfully');
    }

    public function setLead(Request $request, Team $team)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
        ]);

        $team->update(['lead_id' => $validated['user_id']]);

        return $this->success($team->load('lead'), 'Team lead updated successfully');
    }
}
