<?php

namespace App\Http\Controllers\Api;

use App\Models\Vendor;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;

class VendorController extends ApiController
{
    public function index(Request $request): JsonResponse
    {
        $query = Vendor::where('tenant_id', $request->user()->tenant_id);

        if ($request->search) {
            $query->where(function ($q) use ($request) {
                $q->where('name', 'like', "%{$request->search}%")
                  ->orWhere('email', 'like', "%{$request->search}%");
            });
        }

        $vendors = $query->withSum(['bills' => fn($q) => $q->whereIn('status', ['received', 'partial'])], 'total')
            ->orderBy('name')
            ->paginate($request->per_page ?? 50);

        return $this->paginated($vendors);
    }

    public function store(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:50',
            'website' => 'nullable|url|max:255',
            'address_line_1' => 'nullable|string|max:255',
            'address_line_2' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|size:2',
            'tax_number' => 'nullable|string|max:50',
            'currency' => 'nullable|string|size:3',
            'payment_terms' => 'nullable|integer|min:0|max:365',
            'notes' => 'nullable|string',
        ]);

        $vendor = Vendor::create([
            'tenant_id' => $request->user()->tenant_id,
            ...$validated,
            'currency' => $validated['currency'] ?? 'BHD',
        ]);

        return $this->success($vendor, 'Vendor created', 201);
    }

    public function show(Vendor $vendor): JsonResponse
    {
        $this->authorize('view', $vendor);

        return $this->success($vendor->loadCount('bills')
            ->loadSum(['bills' => fn($q) => $q->whereIn('status', ['received', 'partial'])], 'total'));
    }

    public function update(Request $request, Vendor $vendor): JsonResponse
    {
        $this->authorize('update', $vendor);

        $validated = $request->validate([
            'name' => 'sometimes|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:50',
            'website' => 'nullable|url|max:255',
            'address_line_1' => 'nullable|string|max:255',
            'address_line_2' => 'nullable|string|max:255',
            'city' => 'nullable|string|max:100',
            'state' => 'nullable|string|max:100',
            'postal_code' => 'nullable|string|max:20',
            'country' => 'nullable|string|size:2',
            'tax_number' => 'nullable|string|max:50',
            'currency' => 'nullable|string|size:3',
            'payment_terms' => 'nullable|integer|min:0|max:365',
            'notes' => 'nullable|string',
            'is_active' => 'boolean',
        ]);

        $vendor->update($validated);
        return $this->success($vendor, 'Vendor updated');
    }

    public function destroy(Vendor $vendor): JsonResponse
    {
        $this->authorize('delete', $vendor);

        if ($vendor->bills()->exists()) {
            return $this->error('Cannot delete vendor with bills', 422);
        }

        $vendor->delete();
        return $this->success(null, 'Vendor deleted');
    }
}
