<?php

namespace App\Http\Controllers\Api;

use App\Services\TwilioService;
use App\Services\WhatsappService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Response;

class WebhookController extends ApiController
{
    public function stripe(Request $request): Response
    {
        $payload = $request->getContent();
        $sig = $request->header('Stripe-Signature');

        try {
            $event = \Stripe\Webhook::constructEvent(
                $payload,
                $sig,
                config('services.stripe.webhook_secret')
            );

            match ($event->type) {
                'payment_intent.succeeded' => $this->handleStripePayment($event->data->object),
                'invoice.paid' => $this->handleStripeInvoicePaid($event->data->object),
                default => null,
            };

            return response('', 200);
        } catch (\Exception $e) {
            return response('Webhook Error', 400);
        }
    }

    protected function handleStripePayment($paymentIntent): void
    {
        // Handle successful payment
    }

    protected function handleStripeInvoicePaid($invoice): void
    {
        // Handle paid invoice
    }

    public function twilio(Request $request, TwilioService $twilioService): Response
    {
        $data = $request->all();

        if (isset($data['MessageStatus'])) {
            $twilioService->handleWebhook($data);
        } elseif (isset($data['Body'])) {
            $twilioService->handleIncoming($data);
        }

        return response('', 200);
    }

    public function whatsapp(Request $request, WhatsappService $whatsappService): Response
    {
        $whatsappService->handleWebhook($request->all());
        return response('', 200);
    }

    public function verifyWhatsapp(Request $request): Response
    {
        $mode = $request->get('hub_mode');
        $token = $request->get('hub_verify_token');
        $challenge = $request->get('hub_challenge');

        if ($mode === 'subscribe' && $token === config('services.whatsapp.verify_token')) {
            return response($challenge, 200);
        }

        return response('Forbidden', 403);
    }

    public function sendgrid(Request $request): Response
    {
        $events = $request->all();

        foreach ($events as $event) {
            $email = \App\Models\Email::where('message_id', $event['sg_message_id'] ?? null)->first();
            
            if ($email) {
                match ($event['event']) {
                    'delivered' => $email->update(['status' => 'delivered', 'delivered_at' => now()]),
                    'open' => $email->increment('open_count'),
                    'click' => $email->increment('click_count'),
                    'bounce' => $email->update(['status' => 'bounced', 'error' => $event['reason'] ?? null]),
                    default => null,
                };
            }
        }

        return response('', 200);
    }

    public function plaid(Request $request): Response
    {
        $type = $request->input('webhook_type');
        $code = $request->input('webhook_code');
        $itemId = $request->input('item_id');

        $account = \App\Models\BankAccount::where('plaid_item_id', $itemId)->first();

        if ($account) {
            match ("$type.$code") {
                'TRANSACTIONS.INITIAL_UPDATE', 'TRANSACTIONS.HISTORICAL_UPDATE', 'TRANSACTIONS.DEFAULT_UPDATE'
                    => \App\Jobs\SyncBankTransactions::dispatch($account),
                'ITEM.ERROR' => $account->update(['plaid_status' => 'error']),
                default => null,
            };
        }

        return response('', 200);
    }
}
