<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;

class ContactRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        $rules = [
            'type' => 'sometimes|in:person,company',
            'first_name' => 'nullable|string|max:100',
            'last_name' => 'nullable|string|max:100',
            'company_name' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'phone' => 'nullable|string|max:30',
            'mobile' => 'nullable|string|max:30',
            'whatsapp' => 'nullable|string|max:30',
            'website' => 'nullable|url|max:255',
            'address' => 'nullable|string|max:500',
            'city' => 'nullable|string|max:100',
            'country' => 'nullable|string|size:2',
            'cr_number' => 'nullable|string|max:50',
            'nationality' => 'nullable|string|max:100',
            'ownership_type' => 'nullable|in:100_foreign,51_bahraini,gcc_owned,mixed',
            'business_sector' => 'nullable|string|max:100',
            'source' => 'nullable|string|max:100',
            'lead_status' => 'nullable|in:new,contacted,qualified,proposal,negotiation,won,lost',
            'owner_id' => 'nullable|exists:users,id',
            'company_id' => 'nullable|exists:contacts,id',
            'currency' => 'nullable|string|size:3',
            'tags' => 'nullable|array',
            'tags.*' => 'exists:tags,id',
            'custom_fields' => 'nullable|array',
        ];

        if ($this->isMethod('POST')) {
            $rules['type'] = 'required|in:person,company';
        }

        return $rules;
    }
}

class InvoiceRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'contact_id' => 'required|exists:contacts,id',
            'opportunity_id' => 'nullable|exists:opportunities,id',
            'invoice_date' => 'required|date',
            'due_date' => 'required|date|after_or_equal:invoice_date',
            'currency' => 'nullable|string|size:3',
            'notes' => 'nullable|string|max:1000',
            'terms' => 'nullable|string|max:1000',
            'items' => 'required|array|min:1',
            'items.*.id' => 'nullable|exists:invoice_items,id',
            'items.*.product_id' => 'nullable|exists:products,id',
            'items.*.name' => 'required|string|max:255',
            'items.*.description' => 'nullable|string|max:500',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.unit_price' => 'required|numeric|min:0',
            'items.*.discount' => 'nullable|numeric|min:0|max:100',
            'items.*.tax_rate' => 'nullable|numeric|min:0|max:100',
        ];
    }
}

class ProjectRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        $rules = [
            'name' => 'required|string|max:255',
            'code' => 'nullable|string|max:20',
            'description' => 'nullable|string|max:2000',
            'workspace_id' => 'nullable|exists:workspaces,id',
            'contact_id' => 'nullable|exists:contacts,id',
            'color' => 'nullable|string|max:7',
            'status' => 'nullable|in:planning,active,on_hold,completed,cancelled,archived',
            'priority' => 'nullable|in:low,medium,high,urgent',
            'visibility' => 'nullable|in:private,team,public',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'budget' => 'nullable|numeric|min:0',
            'hourly_rate' => 'nullable|numeric|min:0',
            'estimated_hours' => 'nullable|integer|min:0',
            'is_billable' => 'boolean',
        ];

        if ($this->isMethod('PUT')) {
            $rules['name'] = 'sometimes|string|max:255';
        }

        return $rules;
    }
}

class TaskRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        $rules = [
            'project_id' => 'required|exists:projects,id',
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'status_id' => 'nullable|exists:task_statuses,id',
            'parent_id' => 'nullable|exists:tasks,id',
            'group_id' => 'nullable|exists:task_groups,id',
            'assignee_id' => 'nullable|exists:users,id',
            'priority' => 'nullable|in:low,medium,high,urgent',
            'type' => 'nullable|in:task,bug,feature,improvement,milestone,subtask',
            'start_date' => 'nullable|date',
            'due_date' => 'nullable|date|after_or_equal:start_date',
            'estimated_hours' => 'nullable|numeric|min:0',
            'is_billable' => 'boolean',
            'tags' => 'nullable|array',
            'tags.*' => 'exists:tags,id',
        ];

        if ($this->isMethod('PUT')) {
            $rules['project_id'] = 'sometimes|exists:projects,id';
            $rules['title'] = 'sometimes|string|max:255';
        }

        return $rules;
    }
}

class OpportunityRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'title' => $this->isMethod('POST') ? 'required|string|max:255' : 'sometimes|string|max:255',
            'pipeline_id' => $this->isMethod('POST') ? 'required|exists:pipelines,id' : 'sometimes|exists:pipelines,id',
            'stage_id' => $this->isMethod('POST') ? 'required|exists:pipeline_stages,id' : 'sometimes|exists:pipeline_stages,id',
            'contact_id' => 'nullable|exists:contacts,id',
            'company_id' => 'nullable|exists:contacts,id',
            'value' => 'nullable|numeric|min:0',
            'currency' => 'nullable|string|size:3',
            'expected_close_date' => 'nullable|date',
            'description' => 'nullable|string|max:2000',
            'source' => 'nullable|string|max:100',
            'status' => 'nullable|in:open,won,lost',
            'lost_reason' => 'nullable|string|max:500',
            'package' => 'nullable|in:standard,gold,premium',
            'business_activity' => 'nullable|string|max:255',
            'legal_form' => 'nullable|in:WLL,SPC,Branch',
        ];
    }
}

class EmployeeRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        $rules = [
            'user_id' => 'required|exists:users,id',
            'employee_number' => 'nullable|string|max:50',
            'department_id' => 'nullable|exists:teams,id',
            'manager_id' => 'nullable|exists:users,id',
            'position' => 'nullable|string|max:100',
            'employment_type' => 'required|in:full_time,part_time,contract,intern,freelance',
            'salary_type' => 'required|in:monthly,hourly,daily,weekly,annual',
            'salary_amount' => 'required|numeric|min:0',
            'currency' => 'nullable|string|size:3',
            'hire_date' => 'required|date',
            'probation_end_date' => 'nullable|date|after:hire_date',
            'work_location' => 'nullable|string|max:100',
            'bank_name' => 'nullable|string|max:100',
            'bank_account_number' => 'nullable|string|max:50',
            'bank_iban' => 'nullable|string|max:50',
            'cpr_number' => 'nullable|string|max:20',
            'passport_number' => 'nullable|string|max:20',
            'passport_expiry' => 'nullable|date',
            'visa_number' => 'nullable|string|max:50',
            'visa_expiry' => 'nullable|date',
            'emergency_contact_name' => 'nullable|string|max:100',
            'emergency_contact_phone' => 'nullable|string|max:30',
            'emergency_contact_relation' => 'nullable|string|max:50',
        ];

        if ($this->isMethod('POST')) {
            $rules['user_id'] .= '|unique:employees,user_id';
        }

        if ($this->isMethod('PUT')) {
            $rules = array_map(fn($rule) => str_replace('required', 'sometimes', $rule), $rules);
        }

        return $rules;
    }
}

class TimeEntryRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        return [
            'project_id' => 'required|exists:projects,id',
            'task_id' => 'nullable|exists:tasks,id',
            'date' => 'required|date',
            'hours' => 'required|numeric|min:0.01|max:24',
            'start_time' => 'nullable|date_format:H:i',
            'end_time' => 'nullable|date_format:H:i|after:start_time',
            'description' => 'nullable|string|max:500',
            'is_billable' => 'boolean',
        ];
    }
}
