<?php

namespace App\Jobs;

use App\Models\Email;
use App\Models\EmailAccount;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Mail;
use Illuminate\Mail\Message;

class SendEmail implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $backoff = 60;

    public function __construct(
        public Email $email
    ) {}

    public function handle(): void
    {
        $email = $this->email;
        $emailAccount = $email->emailAccount;

        if (!$emailAccount || !$emailAccount->canSend()) {
            $email->markAsFailed('Email account cannot send');
            return;
        }

        // Configure mailer with account settings
        $this->configureMailer($emailAccount);

        try {
            Mail::send([], [], function (Message $message) use ($email) {
                $message->from($email->from_email, $email->from_name)
                    ->subject($email->subject)
                    ->html($email->body_html);

                // Add recipients
                foreach ($email->to as $recipient) {
                    $message->to($recipient);
                }

                if ($email->cc) {
                    foreach ($email->cc as $cc) {
                        $message->cc($cc);
                    }
                }

                if ($email->bcc) {
                    foreach ($email->bcc as $bcc) {
                        $message->bcc($bcc);
                    }
                }

                if ($email->reply_to) {
                    $message->replyTo($email->reply_to);
                }

                // Add attachments
                foreach ($email->attachments as $attachment) {
                    $message->attach(storage_path('app/private/' . $attachment->path), [
                        'as' => $attachment->name,
                        'mime' => $attachment->mime_type,
                    ]);
                }

                // Add tracking pixel
                $trackingPixel = route('tracking.pixel', ['id' => $email->id]);
                $html = $message->getHtmlBody() . "<img src=\"{$trackingPixel}\" width=\"1\" height=\"1\" />";
                $message->html($html);
            });

            // Update email status
            $email->markAsSent();
            
            // Increment account counter
            $emailAccount->incrementSentCount();

        } catch (\Exception $e) {
            $email->markAsFailed($e->getMessage());
            throw $e;
        }
    }

    protected function configureMailer(EmailAccount $account): void
    {
        $config = $account->getSmtpConfig();

        config([
            'mail.mailers.dynamic' => [
                'transport' => 'smtp',
                'host' => $config['host'],
                'port' => $config['port'],
                'encryption' => $config['encryption'],
                'username' => $config['username'],
                'password' => $config['password'],
            ],
        ]);

        Mail::mailer('dynamic');
    }

    public function failed(\Throwable $exception): void
    {
        $this->email->markAsFailed($exception->getMessage());
        
        \Log::error('Failed to send email', [
            'email_id' => $this->email->id,
            'error' => $exception->getMessage(),
        ]);
    }
}
