<?php

namespace App\Jobs;

use App\Models\Invoice;
use App\Notifications\InvoiceReminderNotification;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;

class SendInvoiceReminder implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;

    public function __construct(
        public Invoice $invoice,
        public string $reminderType = 'due'
    ) {}

    public function handle(): void
    {
        $contact = $this->invoice->contact;

        if (!$contact || !$contact->email) {
            return;
        }

        // Send notification
        $contact->notify(new InvoiceReminderNotification($this->invoice, $this->reminderType));

        // Update invoice
        $reminders = $this->invoice->reminders_sent ?? [];
        $reminders[] = [
            'type' => $this->reminderType,
            'sent_at' => now()->toIso8601String(),
        ];
        
        $this->invoice->update([
            'reminders_sent' => $reminders,
            'last_reminder_at' => now(),
        ]);
    }
}
