<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;

class BankReconciliation extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'bank_account_id',
        'statement_date',
        'statement_balance',
        'starting_balance',
        'cleared_balance',
        'difference',
        'status',
        'started_at',
        'completed_at',
        'completed_by',
        'notes',
    ];

    protected $casts = [
        'statement_date' => 'date',
        'statement_balance' => 'decimal:3',
        'starting_balance' => 'decimal:3',
        'cleared_balance' => 'decimal:3',
        'difference' => 'decimal:3',
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
    ];

    const STATUS_IN_PROGRESS = 'in_progress';
    const STATUS_COMPLETED = 'completed';
    const STATUS_CANCELLED = 'cancelled';

    public function bankAccount(): BelongsTo
    {
        return $this->belongsTo(BankAccount::class);
    }

    public function completedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'completed_by');
    }

    public function calculateClearedBalance(): float
    {
        $credits = $this->bankAccount->transactions()
            ->where('is_reconciled', true)
            ->where('type', 'credit')
            ->where('transaction_date', '<=', $this->statement_date)
            ->sum('amount');

        $debits = $this->bankAccount->transactions()
            ->where('is_reconciled', true)
            ->where('type', 'debit')
            ->where('transaction_date', '<=', $this->statement_date)
            ->sum('amount');

        return $this->starting_balance + $credits - $debits;
    }

    public function recalculate(): void
    {
        $this->cleared_balance = $this->calculateClearedBalance();
        $this->difference = $this->statement_balance - $this->cleared_balance;
        $this->save();
    }

    public function isBalanced(): bool
    {
        return abs($this->difference) < 0.01;
    }

    public function complete(User $user): bool
    {
        if (!$this->isBalanced()) {
            return false;
        }

        $this->update([
            'status' => self::STATUS_COMPLETED,
            'completed_at' => now(),
            'completed_by' => $user->id,
        ]);

        $this->bankAccount->updateBalance();

        return true;
    }

    public function cancel(): void
    {
        // Unreconcile all transactions for this period
        $this->bankAccount->transactions()
            ->where('reconciled_at', '>=', $this->started_at)
            ->update([
                'is_reconciled' => false,
                'reconciled_at' => null,
            ]);

        $this->update(['status' => self::STATUS_CANCELLED]);
    }
}
