<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;

class Campaign extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'name',
        'type',
        'status',
        'subject',
        'content',
        'from_name',
        'from_email',
        'reply_to',
        'template_id',
        'email_account_id',
        'audience_filters',
        'settings',
        'recipient_count',
        'sent_count',
        'delivered_count',
        'opened_count',
        'clicked_count',
        'bounced_count',
        'unsubscribed_count',
        'scheduled_at',
        'started_at',
        'completed_at',
        'created_by',
    ];

    protected $casts = [
        'audience_filters' => 'array',
        'settings' => 'array',
        'recipient_count' => 'integer',
        'sent_count' => 'integer',
        'delivered_count' => 'integer',
        'opened_count' => 'integer',
        'clicked_count' => 'integer',
        'bounced_count' => 'integer',
        'unsubscribed_count' => 'integer',
        'scheduled_at' => 'datetime',
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
    ];

    const TYPE_EMAIL = 'email';
    const TYPE_SMS = 'sms';
    const TYPE_WHATSAPP = 'whatsapp';

    const STATUS_DRAFT = 'draft';
    const STATUS_SCHEDULED = 'scheduled';
    const STATUS_SENDING = 'sending';
    const STATUS_SENT = 'sent';
    const STATUS_PAUSED = 'paused';
    const STATUS_CANCELLED = 'cancelled';

    public function template(): BelongsTo
    {
        return $this->belongsTo(EmailTemplate::class);
    }

    public function emailAccount(): BelongsTo
    {
        return $this->belongsTo(EmailAccount::class);
    }

    public function recipients(): HasMany
    {
        return $this->hasMany(CampaignRecipient::class);
    }

    public function emails(): HasMany
    {
        return $this->hasMany(Email::class);
    }

    public function smsMessages(): HasMany
    {
        return $this->hasMany(SmsMessage::class);
    }

    public function whatsappMessages(): HasMany
    {
        return $this->hasMany(WhatsappMessage::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function schedule(\DateTime $dateTime): void
    {
        $this->update([
            'status' => self::STATUS_SCHEDULED,
            'scheduled_at' => $dateTime,
        ]);
    }

    public function send(): void
    {
        $this->update([
            'status' => self::STATUS_SENDING,
            'started_at' => now(),
        ]);

        // Dispatch job to process sending
    }

    public function pause(): void
    {
        $this->update(['status' => self::STATUS_PAUSED]);
    }

    public function resume(): void
    {
        $this->update(['status' => self::STATUS_SENDING]);
    }

    public function cancel(): void
    {
        $this->update(['status' => self::STATUS_CANCELLED]);
    }

    public function complete(): void
    {
        $this->update([
            'status' => self::STATUS_SENT,
            'completed_at' => now(),
        ]);
    }

    public function getOpenRate(): float
    {
        if ($this->delivered_count == 0) {
            return 0;
        }

        return ($this->opened_count / $this->delivered_count) * 100;
    }

    public function getClickRate(): float
    {
        if ($this->opened_count == 0) {
            return 0;
        }

        return ($this->clicked_count / $this->opened_count) * 100;
    }

    public function getBounceRate(): float
    {
        if ($this->sent_count == 0) {
            return 0;
        }

        return ($this->bounced_count / $this->sent_count) * 100;
    }

    public function getAudienceCount(): int
    {
        if (empty($this->audience_filters)) {
            return Contact::where('tenant_id', $this->tenant_id)->count();
        }

        // Apply filters and count
        return Contact::where('tenant_id', $this->tenant_id)
            ->filter($this->audience_filters)
            ->count();
    }

    public function duplicate(): self
    {
        $copy = $this->replicate([
            'status', 'recipient_count', 'sent_count', 'delivered_count',
            'opened_count', 'clicked_count', 'bounced_count', 'unsubscribed_count',
            'scheduled_at', 'started_at', 'completed_at',
        ]);
        $copy->name = $this->name . ' (Copy)';
        $copy->status = self::STATUS_DRAFT;
        $copy->save();

        return $copy;
    }
}
