<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use App\Traits\BelongsToTenant;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class Contact extends Model
{
    use HasFactory, SoftDeletes, BelongsToTenant, LogsActivity;

    protected $fillable = [
        'tenant_id',
        'owner_id',
        'type',
        'company_id',
        'first_name',
        'last_name',
        'company_name',
        'job_title',
        'email',
        'phone',
        'mobile',
        'whatsapp',
        'website',
        'address',
        'city',
        'state',
        'postal_code',
        'country',
        'linkedin',
        'twitter',
        'facebook',
        'cr_number',
        'nationality',
        'ownership_type',
        'business_sector',
        'source',
        'lead_status',
        'lead_score',
        'currency',
        'lifetime_value',
        'date_of_birth',
        'last_contacted_at',
        'converted_at',
        'status',
        'do_not_contact',
        'email_opt_out',
        'sms_opt_out',
        'custom_fields',
        'notes',
    ];

    protected $casts = [
        'custom_fields' => 'array',
        'lead_score' => 'decimal:2',
        'lifetime_value' => 'decimal:2',
        'date_of_birth' => 'date',
        'last_contacted_at' => 'datetime',
        'converted_at' => 'datetime',
        'do_not_contact' => 'boolean',
        'email_opt_out' => 'boolean',
        'sms_opt_out' => 'boolean',
    ];

    protected $appends = ['full_name', 'display_name'];

    // Activity Log
    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['first_name', 'last_name', 'email', 'phone', 'lead_status', 'owner_id'])
            ->logOnlyDirty();
    }

    // Relationships
    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class);
    }

    public function owner(): BelongsTo
    {
        return $this->belongsTo(User::class, 'owner_id');
    }

    public function company(): BelongsTo
    {
        return $this->belongsTo(Contact::class, 'company_id');
    }

    public function employees(): HasMany
    {
        return $this->hasMany(Contact::class, 'company_id');
    }

    public function tags(): BelongsToMany
    {
        return $this->belongsToMany(Tag::class, 'contact_tag');
    }

    public function opportunities(): HasMany
    {
        return $this->hasMany(Opportunity::class);
    }

    public function invoices(): HasMany
    {
        return $this->hasMany(Invoice::class);
    }

    public function activities(): MorphMany
    {
        return $this->morphMany(Activity::class, 'activitable');
    }

    public function conversations(): HasMany
    {
        return $this->hasMany(Conversation::class);
    }

    public function projects(): HasMany
    {
        return $this->hasMany(Project::class);
    }

    // Accessors
    public function getFullNameAttribute(): string
    {
        return trim("{$this->first_name} {$this->last_name}");
    }

    public function getDisplayNameAttribute(): string
    {
        if ($this->type === 'company') {
            return $this->company_name ?? $this->full_name;
        }
        
        return $this->full_name ?: $this->email;
    }

    // Scopes
    public function scopePeople($query)
    {
        return $query->where('type', 'person');
    }

    public function scopeCompanies($query)
    {
        return $query->where('type', 'company');
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeLeads($query)
    {
        return $query->whereNotNull('lead_status')
            ->whereNotIn('lead_status', ['won', 'lost']);
    }

    // Helpers
    public function canContact(): bool
    {
        return !$this->do_not_contact;
    }

    public function canEmail(): bool
    {
        return $this->email && !$this->email_opt_out && !$this->do_not_contact;
    }

    public function canSms(): bool
    {
        return ($this->phone || $this->mobile) && !$this->sms_opt_out && !$this->do_not_contact;
    }

    public function updateLastContacted(): void
    {
        $this->update(['last_contacted_at' => now()]);
    }

    public function addToLifetimeValue(float $amount): void
    {
        $this->increment('lifetime_value', $amount);
    }
}
