<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\HasMany;

class FiscalYear extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'name',
        'start_date',
        'end_date',
        'status',
    ];

    protected $casts = [
        'start_date' => 'date',
        'end_date' => 'date',
    ];

    const STATUS_OPEN = 'open';
    const STATUS_CLOSED = 'closed';

    public function journalEntries(): HasMany
    {
        return $this->hasMany(JournalEntry::class);
    }

    public function isOpen(): bool
    {
        return $this->status === self::STATUS_OPEN;
    }

    public function isClosed(): bool
    {
        return $this->status === self::STATUS_CLOSED;
    }

    public function close(): void
    {
        $this->update(['status' => self::STATUS_CLOSED]);
    }

    public function reopen(): void
    {
        $this->update(['status' => self::STATUS_OPEN]);
    }

    public function containsDate($date): bool
    {
        $date = \Carbon\Carbon::parse($date);
        return $date->between($this->start_date, $this->end_date);
    }

    public static function current($tenantId): ?self
    {
        return static::where('tenant_id', $tenantId)
            ->where('status', self::STATUS_OPEN)
            ->where('start_date', '<=', now())
            ->where('end_date', '>=', now())
            ->first();
    }

    public static function forDate($tenantId, $date): ?self
    {
        return static::where('tenant_id', $tenantId)
            ->where('start_date', '<=', $date)
            ->where('end_date', '>=', $date)
            ->first();
    }

    public static function createNext($tenantId): self
    {
        $latest = static::where('tenant_id', $tenantId)
            ->orderBy('end_date', 'desc')
            ->first();

        if ($latest) {
            $startDate = $latest->end_date->addDay();
            $endDate = $startDate->copy()->addYear()->subDay();
        } else {
            $startDate = now()->startOfYear();
            $endDate = now()->endOfYear();
        }

        return static::create([
            'tenant_id' => $tenantId,
            'name' => 'FY ' . $startDate->format('Y'),
            'start_date' => $startDate,
            'end_date' => $endDate,
            'status' => self::STATUS_OPEN,
        ]);
    }
}
