<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Str;

class Form extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'name',
        'slug',
        'description',
        'fields',
        'settings',
        'style',
        'redirect_url',
        'thank_you_message',
        'workflow_id',
        'is_active',
        'submission_count',
        'created_by',
    ];

    protected $casts = [
        'fields' => 'array',
        'settings' => 'array',
        'style' => 'array',
        'is_active' => 'boolean',
        'submission_count' => 'integer',
    ];

    protected static function booted(): void
    {
        static::creating(function ($form) {
            if (empty($form->slug)) {
                $form->slug = Str::slug($form->name) . '-' . Str::random(6);
            }
        });
    }

    public function workflow(): BelongsTo
    {
        return $this->belongsTo(Workflow::class);
    }

    public function submissions(): HasMany
    {
        return $this->hasMany(FormSubmission::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function publish(): void
    {
        $this->update(['is_active' => true]);
    }

    public function unpublish(): void
    {
        $this->update(['is_active' => false]);
    }

    public function getUrl(): string
    {
        return url("/f/{$this->slug}");
    }

    public function getEmbedCode(): string
    {
        $url = $this->getUrl();
        
        return <<<HTML
<iframe src="{$url}" width="100%" height="500" frameborder="0"></iframe>
HTML;
    }

    public function getFieldNames(): array
    {
        return collect($this->fields ?? [])
            ->pluck('name')
            ->filter()
            ->toArray();
    }

    public function getRequiredFields(): array
    {
        return collect($this->fields ?? [])
            ->where('required', true)
            ->pluck('name')
            ->toArray();
    }

    public function validateSubmission(array $data): array
    {
        $errors = [];
        
        foreach ($this->fields ?? [] as $field) {
            $name = $field['name'] ?? null;
            $required = $field['required'] ?? false;
            $type = $field['type'] ?? 'text';
            
            if (!$name) continue;
            
            $value = $data[$name] ?? null;
            
            if ($required && empty($value)) {
                $errors[$name] = "The {$field['label']} field is required.";
                continue;
            }
            
            if (!empty($value)) {
                // Type-specific validation
                if ($type === 'email' && !filter_var($value, FILTER_VALIDATE_EMAIL)) {
                    $errors[$name] = "Please enter a valid email address.";
                }
                
                if ($type === 'phone' && !preg_match('/^[\d\s\+\-\(\)]+$/', $value)) {
                    $errors[$name] = "Please enter a valid phone number.";
                }
            }
        }
        
        return $errors;
    }

    public function duplicate(): self
    {
        $copy = $this->replicate(['slug', 'submission_count']);
        $copy->name = $this->name . ' (Copy)';
        $copy->slug = Str::slug($copy->name) . '-' . Str::random(6);
        $copy->submission_count = 0;
        $copy->save();

        return $copy;
    }
}
