<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Traits\BelongsToTenant;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class Invoice extends Model
{
    use HasFactory, SoftDeletes, BelongsToTenant, LogsActivity;

    protected $fillable = [
        'tenant_id',
        'contact_id',
        'opportunity_id',
        'invoice_number',
        'invoice_date',
        'due_date',
        'status',
        'currency',
        'exchange_rate',
        'subtotal',
        'discount_amount',
        'tax_amount',
        'total',
        'amount_paid',
        'amount_due',
        'notes',
        'terms',
        'footer',
        'billing_name',
        'billing_address',
        'billing_email',
        'billing_phone',
        'ar_account_id',
        'revenue_account_id',
        'sent_at',
        'viewed_at',
        'paid_at',
        'created_by',
        'custom_fields',
    ];

    protected $casts = [
        'invoice_date' => 'date',
        'due_date' => 'date',
        'exchange_rate' => 'decimal:6',
        'subtotal' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'tax_amount' => 'decimal:2',
        'total' => 'decimal:2',
        'amount_paid' => 'decimal:2',
        'amount_due' => 'decimal:2',
        'sent_at' => 'datetime',
        'viewed_at' => 'datetime',
        'paid_at' => 'datetime',
        'custom_fields' => 'array',
    ];

    protected static function booted(): void
    {
        static::creating(function (Invoice $invoice) {
            if (!$invoice->invoice_number) {
                $invoice->invoice_number = $invoice->generateInvoiceNumber();
            }
        });
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['status', 'amount_paid', 'total'])
            ->logOnlyDirty();
    }

    // Relationships
    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class);
    }

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function opportunity(): BelongsTo
    {
        return $this->belongsTo(Opportunity::class);
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function arAccount(): BelongsTo
    {
        return $this->belongsTo(ChartOfAccount::class, 'ar_account_id');
    }

    public function revenueAccount(): BelongsTo
    {
        return $this->belongsTo(ChartOfAccount::class, 'revenue_account_id');
    }

    public function items(): HasMany
    {
        return $this->hasMany(InvoiceItem::class);
    }

    public function payments(): HasMany
    {
        return $this->hasMany(Payment::class, 'payable_id')
            ->where('payable_type', self::class);
    }

    public function timeEntries(): HasMany
    {
        return $this->hasMany(TimeEntry::class);
    }

    // Scopes
    public function scopeDraft($query)
    {
        return $query->where('status', 'draft');
    }

    public function scopeSent($query)
    {
        return $query->where('status', 'sent');
    }

    public function scopePaid($query)
    {
        return $query->where('status', 'paid');
    }

    public function scopeOverdue($query)
    {
        return $query->where('status', 'overdue')
            ->orWhere(function ($q) {
                $q->whereIn('status', ['sent', 'partial'])
                    ->where('due_date', '<', now());
            });
    }

    public function scopeUnpaid($query)
    {
        return $query->whereIn('status', ['sent', 'partial', 'overdue']);
    }

    // Helpers
    public function generateInvoiceNumber(): string
    {
        $prefix = 'INV-';
        $year = now()->format('Y');
        
        $lastInvoice = static::where('tenant_id', $this->tenant_id)
            ->where('invoice_number', 'like', "{$prefix}{$year}%")
            ->orderBy('invoice_number', 'desc')
            ->first();

        if ($lastInvoice) {
            $lastNumber = (int) substr($lastInvoice->invoice_number, -5);
            $newNumber = $lastNumber + 1;
        } else {
            $newNumber = 1;
        }

        return $prefix . $year . '-' . str_pad($newNumber, 5, '0', STR_PAD_LEFT);
    }

    public function calculateTotals(): void
    {
        $subtotal = 0;
        $taxAmount = 0;

        foreach ($this->items as $item) {
            $itemSubtotal = $item->quantity * $item->unit_price;
            $itemDiscount = $itemSubtotal * ($item->discount_percent / 100);
            $itemNet = $itemSubtotal - $itemDiscount;
            $itemTax = $itemNet * ($item->tax_rate / 100);

            $subtotal += $itemNet;
            $taxAmount += $itemTax;

            $item->update([
                'discount_amount' => $itemDiscount,
                'tax_amount' => $itemTax,
                'total' => $itemNet + $itemTax,
            ]);
        }

        $total = $subtotal + $taxAmount - $this->discount_amount;
        $amountDue = $total - $this->amount_paid;

        $this->update([
            'subtotal' => $subtotal,
            'tax_amount' => $taxAmount,
            'total' => $total,
            'amount_due' => $amountDue,
        ]);
    }

    public function isDraft(): bool
    {
        return $this->status === 'draft';
    }

    public function isPaid(): bool
    {
        return $this->status === 'paid';
    }

    public function isOverdue(): bool
    {
        return $this->due_date->isPast() && $this->amount_due > 0;
    }

    public function markAsSent(): void
    {
        $this->update([
            'status' => 'sent',
            'sent_at' => now(),
        ]);
    }

    public function markAsViewed(): void
    {
        if (!$this->viewed_at) {
            $this->update([
                'status' => $this->status === 'sent' ? 'viewed' : $this->status,
                'viewed_at' => now(),
            ]);
        }
    }

    public function recordPayment(float $amount, array $data = []): Payment
    {
        $payment = $this->payments()->create(array_merge([
            'tenant_id' => $this->tenant_id,
            'type' => 'received',
            'amount' => $amount,
            'currency' => $this->currency,
            'payment_date' => now(),
            'created_by' => auth()->id(),
        ], $data));

        $this->amount_paid += $amount;
        $this->amount_due = $this->total - $this->amount_paid;

        if ($this->amount_due <= 0) {
            $this->status = 'paid';
            $this->paid_at = now();
        } elseif ($this->amount_paid > 0) {
            $this->status = 'partial';
        }

        $this->save();

        // Update contact lifetime value
        if ($this->contact) {
            $this->contact->addToLifetimeValue($amount);
        }

        return $payment;
    }

    public function getFormattedTotal(): string
    {
        return number_format($this->total, 2) . ' ' . $this->currency;
    }

    public function getDaysOverdue(): int
    {
        if (!$this->isOverdue()) {
            return 0;
        }

        return $this->due_date->diffInDays(now());
    }
}
