<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;

class SalaryComponent extends BaseModel
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'tenant_id',
        'name',
        'code',
        'type',
        'category',
        'calculation_type',
        'default_amount',
        'percentage_of',
        'formula',
        'is_taxable',
        'is_active',
        'applies_to_all',
        'description',
    ];

    protected $casts = [
        'default_amount' => 'decimal:3',
        'percentage_of' => 'decimal:2',
        'is_taxable' => 'boolean',
        'is_active' => 'boolean',
        'applies_to_all' => 'boolean',
    ];

    const TYPE_EARNING = 'earning';
    const TYPE_DEDUCTION = 'deduction';
    const TYPE_EMPLOYER_CONTRIBUTION = 'employer_contribution';

    const CATEGORY_BASIC = 'basic_salary';
    const CATEGORY_ALLOWANCE = 'allowance';
    const CATEGORY_BONUS = 'bonus';
    const CATEGORY_OVERTIME = 'overtime';
    const CATEGORY_COMMISSION = 'commission';
    const CATEGORY_TAX = 'tax';
    const CATEGORY_INSURANCE = 'insurance';
    const CATEGORY_LOAN = 'loan';
    const CATEGORY_OTHER = 'other';

    const CALC_FIXED = 'fixed';
    const CALC_PERCENTAGE = 'percentage';
    const CALC_FORMULA = 'formula';

    public function calculate(array $context): float
    {
        return match ($this->calculation_type) {
            self::CALC_FIXED => $this->default_amount ?? 0,
            self::CALC_PERCENTAGE => $this->calculatePercentage($context),
            self::CALC_FORMULA => $this->calculateFormula($context),
            default => 0,
        };
    }

    protected function calculatePercentage(array $context): float
    {
        $base = $context['basic_salary'] ?? 0;
        return $base * (($this->percentage_of ?? 0) / 100);
    }

    protected function calculateFormula(array $context): float
    {
        if (empty($this->formula)) {
            return 0;
        }

        // Simple formula parser - replace variables and evaluate
        $formula = $this->formula;
        foreach ($context as $key => $value) {
            $formula = str_replace('{' . $key . '}', $value, $formula);
        }

        // Safety: only allow basic math operations
        if (preg_match('/^[\d\s\+\-\*\/\(\)\.]+$/', $formula)) {
            try {
                return eval("return {$formula};");
            } catch (\Exception $e) {
                return 0;
            }
        }

        return 0;
    }

    public function scopeEarnings($query)
    {
        return $query->where('type', self::TYPE_EARNING);
    }

    public function scopeDeductions($query)
    {
        return $query->where('type', self::TYPE_DEDUCTION);
    }

    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public static function seedDefaults($tenantId): void
    {
        $components = [
            // Earnings
            ['name' => 'Basic Salary', 'code' => 'BASIC', 'type' => self::TYPE_EARNING, 'category' => self::CATEGORY_BASIC, 'calculation_type' => self::CALC_FIXED, 'is_taxable' => true],
            ['name' => 'Housing Allowance', 'code' => 'HOUSING', 'type' => self::TYPE_EARNING, 'category' => self::CATEGORY_ALLOWANCE, 'calculation_type' => self::CALC_PERCENTAGE, 'percentage_of' => 25, 'is_taxable' => true],
            ['name' => 'Transport Allowance', 'code' => 'TRANSPORT', 'type' => self::TYPE_EARNING, 'category' => self::CATEGORY_ALLOWANCE, 'calculation_type' => self::CALC_FIXED, 'default_amount' => 50, 'is_taxable' => true],
            ['name' => 'Overtime', 'code' => 'OT', 'type' => self::TYPE_EARNING, 'category' => self::CATEGORY_OVERTIME, 'calculation_type' => self::CALC_FORMULA, 'formula' => '{overtime_hours} * ({basic_salary} / 240) * 1.25', 'is_taxable' => true],
            // Deductions
            ['name' => 'GOSI - Employee', 'code' => 'GOSI_EE', 'type' => self::TYPE_DEDUCTION, 'category' => self::CATEGORY_INSURANCE, 'calculation_type' => self::CALC_PERCENTAGE, 'percentage_of' => 7, 'is_taxable' => false],
            ['name' => 'Loan Deduction', 'code' => 'LOAN', 'type' => self::TYPE_DEDUCTION, 'category' => self::CATEGORY_LOAN, 'calculation_type' => self::CALC_FIXED, 'is_taxable' => false],
            // Employer Contributions
            ['name' => 'GOSI - Employer', 'code' => 'GOSI_ER', 'type' => self::TYPE_EMPLOYER_CONTRIBUTION, 'category' => self::CATEGORY_INSURANCE, 'calculation_type' => self::CALC_PERCENTAGE, 'percentage_of' => 12, 'is_taxable' => false],
        ];

        foreach ($components as $component) {
            static::firstOrCreate(
                ['tenant_id' => $tenantId, 'code' => $component['code']],
                array_merge($component, ['tenant_id' => $tenantId, 'is_active' => true, 'applies_to_all' => true])
            );
        }
    }
}
