<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use App\Traits\BelongsToTenant;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class Task extends Model
{
    use HasFactory, SoftDeletes, BelongsToTenant, LogsActivity;

    protected $fillable = [
        'tenant_id',
        'project_id',
        'task_list_id',
        'parent_id',
        'status_id',
        'priority_id',
        'assignee_id',
        'creator_id',
        'title',
        'description',
        'start_date',
        'due_date',
        'completed_at',
        'completed_by',
        'estimated_hours',
        'logged_hours',
        'progress',
        'is_milestone',
        'is_billable',
        'hourly_rate',
        'position',
        'subtask_count',
        'completed_subtask_count',
        'custom_fields',
    ];

    protected $casts = [
        'start_date' => 'date',
        'due_date' => 'date',
        'completed_at' => 'datetime',
        'estimated_hours' => 'decimal:2',
        'logged_hours' => 'decimal:2',
        'progress' => 'decimal:2',
        'hourly_rate' => 'decimal:2',
        'is_milestone' => 'boolean',
        'is_billable' => 'boolean',
        'custom_fields' => 'array',
    ];

    protected static function booted(): void
    {
        static::created(function (Task $task) {
            $task->project->updateProgress();
            
            if ($task->parent_id) {
                $task->parent->updateSubtaskCount();
            }
        });

        static::updated(function (Task $task) {
            $task->project->updateProgress();
            
            if ($task->isDirty('completed_at')) {
                if ($task->parent_id) {
                    $task->parent->updateSubtaskCount();
                }
            }
        });

        static::deleted(function (Task $task) {
            $task->project->updateProgress();
            
            if ($task->parent_id) {
                $task->parent->updateSubtaskCount();
            }
        });
    }

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logOnly(['title', 'status_id', 'assignee_id', 'due_date', 'completed_at'])
            ->logOnlyDirty();
    }

    // Relationships
    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class);
    }

    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    public function taskList(): BelongsTo
    {
        return $this->belongsTo(TaskList::class);
    }

    public function parent(): BelongsTo
    {
        return $this->belongsTo(Task::class, 'parent_id');
    }

    public function subtasks(): HasMany
    {
        return $this->hasMany(Task::class, 'parent_id');
    }

    public function status(): BelongsTo
    {
        return $this->belongsTo(TaskStatus::class, 'status_id');
    }

    public function priority(): BelongsTo
    {
        return $this->belongsTo(TaskPriority::class, 'priority_id');
    }

    public function assignee(): BelongsTo
    {
        return $this->belongsTo(User::class, 'assignee_id');
    }

    public function creator(): BelongsTo
    {
        return $this->belongsTo(User::class, 'creator_id');
    }

    public function completedByUser(): BelongsTo
    {
        return $this->belongsTo(User::class, 'completed_by');
    }

    public function assignees(): BelongsToMany
    {
        return $this->belongsToMany(User::class, 'task_assignees')
            ->withTimestamps();
    }

    public function tags(): BelongsToMany
    {
        return $this->belongsToMany(Tag::class, 'task_tag');
    }

    public function dependencies(): BelongsToMany
    {
        return $this->belongsToMany(Task::class, 'task_dependencies', 'task_id', 'depends_on_id')
            ->withPivot('type', 'lag_days')
            ->withTimestamps();
    }

    public function dependents(): BelongsToMany
    {
        return $this->belongsToMany(Task::class, 'task_dependencies', 'depends_on_id', 'task_id')
            ->withPivot('type', 'lag_days')
            ->withTimestamps();
    }

    public function checklists(): HasMany
    {
        return $this->hasMany(TaskChecklist::class);
    }

    public function comments(): HasMany
    {
        return $this->hasMany(TaskComment::class)->latest();
    }

    public function attachments(): HasMany
    {
        return $this->hasMany(TaskAttachment::class);
    }

    public function timeEntries(): HasMany
    {
        return $this->hasMany(TimeEntry::class);
    }

    public function activities(): HasMany
    {
        return $this->hasMany(TaskActivity::class)->latest();
    }

    // Scopes
    public function scopeRoot($query)
    {
        return $query->whereNull('parent_id');
    }

    public function scopeCompleted($query)
    {
        return $query->whereNotNull('completed_at');
    }

    public function scopeIncomplete($query)
    {
        return $query->whereNull('completed_at');
    }

    public function scopeOverdue($query)
    {
        return $query->whereNull('completed_at')
            ->whereNotNull('due_date')
            ->where('due_date', '<', now());
    }

    public function scopeDueToday($query)
    {
        return $query->whereNull('completed_at')
            ->whereDate('due_date', today());
    }

    public function scopeDueThisWeek($query)
    {
        return $query->whereNull('completed_at')
            ->whereBetween('due_date', [now()->startOfWeek(), now()->endOfWeek()]);
    }

    public function scopeAssignedTo($query, User $user)
    {
        return $query->where('assignee_id', $user->id);
    }

    public function scopeByStatus($query, $statusId)
    {
        return $query->where('status_id', $statusId);
    }

    // Helpers
    public function isCompleted(): bool
    {
        return $this->completed_at !== null;
    }

    public function isOverdue(): bool
    {
        return $this->due_date && $this->due_date->isPast() && !$this->isCompleted();
    }

    public function isDueToday(): bool
    {
        return $this->due_date && $this->due_date->isToday();
    }

    public function isSubtask(): bool
    {
        return $this->parent_id !== null;
    }

    public function hasSubtasks(): bool
    {
        return $this->subtask_count > 0;
    }

    public function isBlocked(): bool
    {
        return $this->dependencies()
            ->whereNull('completed_at')
            ->exists();
    }

    public function complete(): void
    {
        $completedStatus = $this->project->taskStatuses()
            ->where('is_completed', true)
            ->first();

        $this->update([
            'completed_at' => now(),
            'completed_by' => auth()->id(),
            'status_id' => $completedStatus?->id ?? $this->status_id,
            'progress' => 100,
        ]);

        $this->logActivity('completed', 'Marked as completed');
    }

    public function reopen(): void
    {
        $defaultStatus = $this->project->taskStatuses()
            ->where('is_default', true)
            ->first();

        $this->update([
            'completed_at' => null,
            'completed_by' => null,
            'status_id' => $defaultStatus?->id ?? $this->status_id,
        ]);

        $this->logActivity('reopened', 'Reopened task');
    }

    public function updateSubtaskCount(): void
    {
        $total = $this->subtasks()->count();
        $completed = $this->subtasks()->whereNotNull('completed_at')->count();
        
        $this->update([
            'subtask_count' => $total,
            'completed_subtask_count' => $completed,
        ]);
    }

    public function updateLoggedHours(): void
    {
        $hours = $this->timeEntries()->sum('hours');
        $this->update(['logged_hours' => $hours]);
    }

    public function logActivity(string $action, string $description = null, array $metadata = []): void
    {
        $this->activities()->create([
            'user_id' => auth()->id(),
            'action' => $action,
            'metadata' => array_merge($metadata, [
                'description' => $description,
            ]),
        ]);
    }

    public function getChecklistProgress(): array
    {
        $total = 0;
        $completed = 0;

        foreach ($this->checklists as $checklist) {
            $total += $checklist->items->count();
            $completed += $checklist->items->where('is_completed', true)->count();
        }

        return [
            'total' => $total,
            'completed' => $completed,
            'percentage' => $total > 0 ? round(($completed / $total) * 100) : 0,
        ];
    }

    public function getAllAssignees(): \Illuminate\Support\Collection
    {
        $assignees = $this->assignees;
        
        if ($this->assignee && !$assignees->contains('id', $this->assignee_id)) {
            $assignees->prepend($this->assignee);
        }

        return $assignees;
    }
}
