<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class TaskAutomation extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'project_id',
        'name',
        'description',
        'trigger',
        'conditions',
        'actions',
        'is_active',
        'run_count',
        'last_run_at',
        'created_by',
    ];

    protected $casts = [
        'trigger' => 'array',
        'conditions' => 'array',
        'actions' => 'array',
        'is_active' => 'boolean',
        'run_count' => 'integer',
        'last_run_at' => 'datetime',
    ];

    const TRIGGER_TASK_CREATED = 'task_created';
    const TRIGGER_TASK_UPDATED = 'task_updated';
    const TRIGGER_TASK_COMPLETED = 'task_completed';
    const TRIGGER_STATUS_CHANGED = 'status_changed';
    const TRIGGER_ASSIGNEE_CHANGED = 'assignee_changed';
    const TRIGGER_DUE_DATE_APPROACHING = 'due_date_approaching';
    const TRIGGER_DUE_DATE_PASSED = 'due_date_passed';
    const TRIGGER_COMMENT_ADDED = 'comment_added';

    const ACTION_CHANGE_STATUS = 'change_status';
    const ACTION_ASSIGN_USER = 'assign_user';
    const ACTION_ADD_TAG = 'add_tag';
    const ACTION_REMOVE_TAG = 'remove_tag';
    const ACTION_SET_PRIORITY = 'set_priority';
    const ACTION_SEND_NOTIFICATION = 'send_notification';
    const ACTION_CREATE_SUBTASK = 'create_subtask';
    const ACTION_MOVE_TO_PROJECT = 'move_to_project';
    const ACTION_SET_DUE_DATE = 'set_due_date';

    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    public function createdBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function shouldTrigger(string $event, Task $task, array $context = []): bool
    {
        if (!$this->is_active) {
            return false;
        }

        $trigger = $this->trigger;

        if (($trigger['type'] ?? null) !== $event) {
            return false;
        }

        // Check project scope
        if ($this->project_id && $task->project_id !== $this->project_id) {
            return false;
        }

        // Check additional trigger conditions
        if (!empty($trigger['conditions'])) {
            foreach ($trigger['conditions'] as $condition) {
                if (!$this->evaluateCondition($condition, $task, $context)) {
                    return false;
                }
            }
        }

        // Check main conditions
        if (!empty($this->conditions)) {
            foreach ($this->conditions as $condition) {
                if (!$this->evaluateCondition($condition, $task, $context)) {
                    return false;
                }
            }
        }

        return true;
    }

    protected function evaluateCondition(array $condition, Task $task, array $context): bool
    {
        $field = $condition['field'] ?? null;
        $operator = $condition['operator'] ?? 'equals';
        $value = $condition['value'] ?? null;

        $taskValue = match ($field) {
            'status_id' => $task->status_id,
            'priority' => $task->priority,
            'assignee_id' => $task->assignee_id,
            'project_id' => $task->project_id,
            'type' => $task->type,
            'old_status_id' => $context['old_status_id'] ?? null,
            'new_status_id' => $context['new_status_id'] ?? null,
            default => $task->{$field} ?? null,
        };

        return match ($operator) {
            'equals' => $taskValue == $value,
            'not_equals' => $taskValue != $value,
            'in' => in_array($taskValue, (array) $value),
            'not_in' => !in_array($taskValue, (array) $value),
            'is_empty' => empty($taskValue),
            'is_not_empty' => !empty($taskValue),
            default => true,
        };
    }

    public function execute(Task $task, array $context = []): void
    {
        foreach ($this->actions as $action) {
            $this->executeAction($action, $task, $context);
        }

        $this->increment('run_count');
        $this->update(['last_run_at' => now()]);
    }

    protected function executeAction(array $action, Task $task, array $context): void
    {
        $type = $action['type'] ?? null;
        $value = $action['value'] ?? null;

        match ($type) {
            self::ACTION_CHANGE_STATUS => $task->update(['status_id' => $value]),
            self::ACTION_ASSIGN_USER => $task->update(['assignee_id' => $value]),
            self::ACTION_SET_PRIORITY => $task->update(['priority' => $value]),
            self::ACTION_ADD_TAG => $task->tags()->attach($value),
            self::ACTION_REMOVE_TAG => $task->tags()->detach($value),
            self::ACTION_SEND_NOTIFICATION => $this->sendNotification($task, $action),
            self::ACTION_CREATE_SUBTASK => $this->createSubtask($task, $action),
            self::ACTION_SET_DUE_DATE => $this->setDueDate($task, $action),
            default => null,
        };
    }

    protected function sendNotification(Task $task, array $action): void
    {
        // Implementation for sending notification
    }

    protected function createSubtask(Task $task, array $action): void
    {
        $task->subtasks()->create([
            'tenant_id' => $task->tenant_id,
            'project_id' => $task->project_id,
            'title' => $action['title'] ?? 'Subtask',
            'status_id' => $task->project->statuses()->where('is_default', true)->first()?->id,
        ]);
    }

    protected function setDueDate(Task $task, array $action): void
    {
        $offset = $action['offset'] ?? 0;
        $from = $action['from'] ?? 'now';

        $baseDate = match ($from) {
            'now' => now(),
            'task_created' => $task->created_at,
            'task_start' => $task->start_date ?? now(),
            default => now(),
        };

        $task->update(['due_date' => $baseDate->addDays($offset)]);
    }
}
