<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use App\Traits\BelongsToTenant;

class TimeEntry extends Model
{
    use HasFactory, SoftDeletes, BelongsToTenant;

    protected $fillable = [
        'tenant_id',
        'user_id',
        'project_id',
        'task_id',
        'date',
        'start_time',
        'end_time',
        'hours',
        'description',
        'is_billable',
        'hourly_rate',
        'amount',
        'currency',
        'is_running',
        'timer_started_at',
        'invoice_id',
        'is_invoiced',
        'status',
        'approved_by',
        'approved_at',
        'rejection_reason',
    ];

    protected $casts = [
        'date' => 'date',
        'start_time' => 'datetime:H:i',
        'end_time' => 'datetime:H:i',
        'hours' => 'decimal:2',
        'hourly_rate' => 'decimal:2',
        'amount' => 'decimal:2',
        'is_billable' => 'boolean',
        'is_running' => 'boolean',
        'timer_started_at' => 'datetime',
        'is_invoiced' => 'boolean',
        'approved_at' => 'datetime',
    ];

    protected static function booted(): void
    {
        static::creating(function (TimeEntry $entry) {
            $entry->calculateAmount();
        });

        static::updating(function (TimeEntry $entry) {
            if ($entry->isDirty(['hours', 'hourly_rate', 'is_billable'])) {
                $entry->calculateAmount();
            }
        });

        static::saved(function (TimeEntry $entry) {
            if ($entry->task) {
                $entry->task->updateLoggedHours();
            }
        });

        static::deleted(function (TimeEntry $entry) {
            if ($entry->task) {
                $entry->task->updateLoggedHours();
            }
        });
    }

    // Relationships
    public function tenant(): BelongsTo
    {
        return $this->belongsTo(Tenant::class);
    }

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function project(): BelongsTo
    {
        return $this->belongsTo(Project::class);
    }

    public function task(): BelongsTo
    {
        return $this->belongsTo(Task::class);
    }

    public function invoice(): BelongsTo
    {
        return $this->belongsTo(Invoice::class);
    }

    public function approver(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    // Scopes
    public function scopeBillable($query)
    {
        return $query->where('is_billable', true);
    }

    public function scopeNonBillable($query)
    {
        return $query->where('is_billable', false);
    }

    public function scopeInvoiced($query)
    {
        return $query->where('is_invoiced', true);
    }

    public function scopeUninvoiced($query)
    {
        return $query->where('is_invoiced', false)
            ->where('is_billable', true);
    }

    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    public function scopePending($query)
    {
        return $query->where('status', 'submitted');
    }

    public function scopeForDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('date', [$startDate, $endDate]);
    }

    public function scopeForUser($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    public function scopeRunning($query)
    {
        return $query->where('is_running', true);
    }

    // Helpers
    public function calculateAmount(): void
    {
        if (!$this->is_billable) {
            $this->amount = 0;
            return;
        }

        $rate = $this->hourly_rate 
            ?? $this->task?->hourly_rate 
            ?? $this->project?->hourly_rate 
            ?? $this->user?->hourly_rate 
            ?? 0;

        $this->hourly_rate = $rate;
        $this->amount = $this->hours * $rate;
    }

    public function startTimer(): void
    {
        $this->update([
            'is_running' => true,
            'timer_started_at' => now(),
        ]);
    }

    public function stopTimer(): void
    {
        if (!$this->is_running || !$this->timer_started_at) {
            return;
        }

        $duration = now()->diffInMinutes($this->timer_started_at);
        $hours = round($duration / 60, 2);

        $this->update([
            'is_running' => false,
            'hours' => $this->hours + $hours,
            'timer_started_at' => null,
        ]);
    }

    public function getRunningDuration(): int
    {
        if (!$this->is_running || !$this->timer_started_at) {
            return 0;
        }

        return now()->diffInMinutes($this->timer_started_at);
    }

    public function submit(): void
    {
        $this->update(['status' => 'submitted']);
    }

    public function approve(User $approver): void
    {
        $this->update([
            'status' => 'approved',
            'approved_by' => $approver->id,
            'approved_at' => now(),
        ]);
    }

    public function reject(User $approver, string $reason = null): void
    {
        $this->update([
            'status' => 'rejected',
            'approved_by' => $approver->id,
            'approved_at' => now(),
            'rejection_reason' => $reason,
        ]);
    }

    public function markAsInvoiced(Invoice $invoice): void
    {
        $this->update([
            'is_invoiced' => true,
            'invoice_id' => $invoice->id,
        ]);
    }

    public function getFormattedDuration(): string
    {
        $hours = floor($this->hours);
        $minutes = round(($this->hours - $hours) * 60);

        if ($hours > 0 && $minutes > 0) {
            return "{$hours}h {$minutes}m";
        } elseif ($hours > 0) {
            return "{$hours}h";
        } else {
            return "{$minutes}m";
        }
    }
}
