<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Timesheet extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'user_id',
        'week_start',
        'week_end',
        'total_hours',
        'billable_hours',
        'status',
        'submitted_at',
        'approved_at',
        'approved_by',
        'rejected_at',
        'rejected_by',
        'rejection_reason',
        'notes',
    ];

    protected $casts = [
        'week_start' => 'date',
        'week_end' => 'date',
        'total_hours' => 'decimal:2',
        'billable_hours' => 'decimal:2',
        'submitted_at' => 'datetime',
        'approved_at' => 'datetime',
        'rejected_at' => 'datetime',
    ];

    const STATUS_DRAFT = 'draft';
    const STATUS_SUBMITTED = 'submitted';
    const STATUS_APPROVED = 'approved';
    const STATUS_REJECTED = 'rejected';

    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    public function approvedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'approved_by');
    }

    public function rejectedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'rejected_by');
    }

    public function timeEntries(): HasMany
    {
        return $this->hasMany(TimeEntry::class);
    }

    public function recalculate(): void
    {
        $entries = TimeEntry::where('user_id', $this->user_id)
            ->whereBetween('date', [$this->week_start, $this->week_end])
            ->get();

        $this->total_hours = $entries->sum('hours');
        $this->billable_hours = $entries->where('is_billable', true)->sum('hours');
        $this->save();
    }

    public function submit(): void
    {
        $this->recalculate();
        $this->update([
            'status' => self::STATUS_SUBMITTED,
            'submitted_at' => now(),
        ]);
    }

    public function approve(User $user): void
    {
        $this->update([
            'status' => self::STATUS_APPROVED,
            'approved_at' => now(),
            'approved_by' => $user->id,
        ]);

        // Update time entries status
        $this->timeEntries()->update(['status' => 'approved']);
    }

    public function reject(User $user, string $reason = null): void
    {
        $this->update([
            'status' => self::STATUS_REJECTED,
            'rejected_at' => now(),
            'rejected_by' => $user->id,
            'rejection_reason' => $reason,
        ]);

        // Update time entries status
        $this->timeEntries()->update(['status' => 'rejected']);
    }

    public function reopen(): void
    {
        $this->update([
            'status' => self::STATUS_DRAFT,
            'submitted_at' => null,
            'approved_at' => null,
            'approved_by' => null,
            'rejected_at' => null,
            'rejected_by' => null,
            'rejection_reason' => null,
        ]);

        $this->timeEntries()->update(['status' => 'draft']);
    }

    public static function getOrCreateForWeek($tenantId, $userId, $date = null): self
    {
        $date = $date ? \Carbon\Carbon::parse($date) : now();
        $weekStart = $date->startOfWeek();
        $weekEnd = $date->endOfWeek();

        return static::firstOrCreate(
            [
                'tenant_id' => $tenantId,
                'user_id' => $userId,
                'week_start' => $weekStart,
            ],
            [
                'week_end' => $weekEnd,
                'total_hours' => 0,
                'billable_hours' => 0,
                'status' => self::STATUS_DRAFT,
            ]
        );
    }

    public function getEntriesByDay(): array
    {
        $entries = TimeEntry::where('user_id', $this->user_id)
            ->whereBetween('date', [$this->week_start, $this->week_end])
            ->get()
            ->groupBy(fn ($entry) => $entry->date->format('Y-m-d'));

        return $entries->toArray();
    }
}
