<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class WorkflowEnrollment extends BaseModel
{
    use HasFactory;

    protected $fillable = [
        'tenant_id',
        'workflow_id',
        'contact_id',
        'current_step',
        'status',
        'next_action_at',
        'step_data',
        'started_at',
        'completed_at',
        'failed_at',
        'error_message',
    ];

    protected $casts = [
        'current_step' => 'integer',
        'step_data' => 'array',
        'next_action_at' => 'datetime',
        'started_at' => 'datetime',
        'completed_at' => 'datetime',
        'failed_at' => 'datetime',
    ];

    const STATUS_ACTIVE = 'active';
    const STATUS_PAUSED = 'paused';
    const STATUS_COMPLETED = 'completed';
    const STATUS_FAILED = 'failed';
    const STATUS_CANCELLED = 'cancelled';

    protected static function booted(): void
    {
        static::creating(function ($enrollment) {
            if (empty($enrollment->started_at)) {
                $enrollment->started_at = now();
            }
            if (empty($enrollment->status)) {
                $enrollment->status = self::STATUS_ACTIVE;
            }
        });
    }

    public function workflow(): BelongsTo
    {
        return $this->belongsTo(Workflow::class);
    }

    public function contact(): BelongsTo
    {
        return $this->belongsTo(Contact::class);
    }

    public function advance(): void
    {
        $this->increment('current_step');
        $this->update(['step_data' => []]);
    }

    public function complete(): void
    {
        $this->update([
            'status' => self::STATUS_COMPLETED,
            'completed_at' => now(),
        ]);
    }

    public function fail(string $reason = null): void
    {
        $this->update([
            'status' => self::STATUS_FAILED,
            'failed_at' => now(),
            'error_message' => $reason,
        ]);
    }

    public function pause(): void
    {
        $this->update(['status' => self::STATUS_PAUSED]);
    }

    public function resume(): void
    {
        $this->update(['status' => self::STATUS_ACTIVE]);
    }

    public function cancel(): void
    {
        $this->update(['status' => self::STATUS_CANCELLED]);
    }

    public function scheduleNextAction(\DateTime $dateTime): void
    {
        $this->update(['next_action_at' => $dateTime]);
    }

    public function getCurrentStepConfig(): ?array
    {
        $steps = $this->workflow->steps ?? [];
        return $steps[$this->current_step] ?? null;
    }

    public function isActive(): bool
    {
        return $this->status === self::STATUS_ACTIVE;
    }

    public function scopeActive($query)
    {
        return $query->where('status', self::STATUS_ACTIVE);
    }

    public function scopeDue($query)
    {
        return $query->where('status', self::STATUS_ACTIVE)
            ->where('next_action_at', '<=', now());
    }
}
