<?php

namespace App\Notifications;

use App\Models\Invoice;
use App\Models\Payment;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class PaymentReceivedNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(public Invoice $invoice, public Payment $payment) {}

    public function via($notifiable): array
    {
        return ['mail', 'database'];
    }

    public function toMail($notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Payment Received for Invoice #' . $this->invoice->invoice_number)
            ->line('A payment has been received.')
            ->line('Invoice: ' . $this->invoice->invoice_number)
            ->line('Amount: ' . number_format($this->payment->amount, 3) . ' ' . $this->payment->currency)
            ->action('View Payment', url('/payments/' . $this->payment->id));
    }

    public function toArray($notifiable): array
    {
        return [
            'type' => 'payment_received',
            'invoice_id' => $this->invoice->id,
            'payment_id' => $this->payment->id,
            'amount' => $this->payment->amount,
        ];
    }
}
