<?php

namespace App\Notifications;

use App\Models\Task;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class TaskOverdueNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(public Task $task) {}

    public function via($notifiable): array
    {
        return ['mail', 'database'];
    }

    public function toMail($notifiable): MailMessage
    {
        $daysOverdue = now()->diffInDays($this->task->due_date);

        return (new MailMessage)
            ->subject("Task Overdue: {$this->task->title}")
            ->greeting("Hello {$notifiable->first_name},")
            ->line("The following task is now {$daysOverdue} days overdue:")
            ->line("Task: {$this->task->title}")
            ->line("Project: {$this->task->project->name}")
            ->line("Original Due Date: " . $this->task->due_date->format('M d, Y'))
            ->action('View Task', url("/tasks/{$this->task->id}"))
            ->line('Please complete this task as soon as possible.');
    }

    public function toArray($notifiable): array
    {
        return [
            'type' => 'task_overdue',
            'task_id' => $this->task->id,
            'title' => $this->task->title,
            'project' => $this->task->project->name,
            'days_overdue' => now()->diffInDays($this->task->due_date),
        ];
    }
}
