<?php

namespace App\Services;

use App\Models\Contact;
use App\Models\Export;
use App\Models\Invoice;
use Illuminate\Support\Facades\Storage;
use League\Csv\Writer;

class ExportService
{
    public function export(int $tenantId, string $type, array $filters, int $userId): Export
    {
        $export = Export::create([
            'tenant_id' => $tenantId,
            'type' => $type,
            'filters' => $filters,
            'status' => 'processing',
            'created_by' => $userId,
        ]);

        try {
            $filePath = match ($type) {
                'contacts' => $this->exportContacts($tenantId, $filters),
                'invoices' => $this->exportInvoices($tenantId, $filters),
                default => throw new \Exception("Unknown export type: {$type}"),
            };

            $export->update([
                'status' => 'completed',
                'file_path' => $filePath,
                'completed_at' => now(),
            ]);
        } catch (\Exception $e) {
            $export->update([
                'status' => 'failed',
                'error_message' => $e->getMessage(),
            ]);
        }

        return $export;
    }

    protected function exportContacts(int $tenantId, array $filters): string
    {
        $query = Contact::where('tenant_id', $tenantId);

        if (!empty($filters['tags'])) {
            $query->whereHas('tags', fn($q) => $q->whereIn('tags.id', $filters['tags']));
        }

        if (!empty($filters['lead_status'])) {
            $query->where('lead_status', $filters['lead_status']);
        }

        $contacts = $query->get();

        $csv = Writer::createFromString();
        $csv->insertOne(['First Name', 'Last Name', 'Email', 'Phone', 'Company', 'Job Title', 'Source', 'Status']);

        foreach ($contacts as $contact) {
            $csv->insertOne([
                $contact->first_name,
                $contact->last_name,
                $contact->email,
                $contact->phone,
                $contact->company_name,
                $contact->job_title,
                $contact->source,
                $contact->lead_status,
            ]);
        }

        $filePath = "exports/contacts_{$tenantId}_" . now()->format('Ymd_His') . ".csv";
        Storage::put($filePath, $csv->toString());

        return $filePath;
    }

    protected function exportInvoices(int $tenantId, array $filters): string
    {
        $query = Invoice::where('tenant_id', $tenantId)->with('contact');

        if (!empty($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        if (!empty($filters['date_from'])) {
            $query->whereDate('invoice_date', '>=', $filters['date_from']);
        }

        if (!empty($filters['date_to'])) {
            $query->whereDate('invoice_date', '<=', $filters['date_to']);
        }

        $invoices = $query->get();

        $csv = Writer::createFromString();
        $csv->insertOne(['Invoice #', 'Date', 'Due Date', 'Customer', 'Subtotal', 'Tax', 'Total', 'Paid', 'Status']);

        foreach ($invoices as $invoice) {
            $csv->insertOne([
                $invoice->invoice_number,
                $invoice->invoice_date->format('Y-m-d'),
                $invoice->due_date->format('Y-m-d'),
                $invoice->contact->full_name,
                $invoice->subtotal,
                $invoice->tax_amount,
                $invoice->total,
                $invoice->amount_paid,
                $invoice->status,
            ]);
        }

        $filePath = "exports/invoices_{$tenantId}_" . now()->format('Ymd_His') . ".csv";
        Storage::put($filePath, $csv->toString());

        return $filePath;
    }

    public function getDownloadUrl(Export $export): string
    {
        return Storage::temporaryUrl($export->file_path, now()->addHours(24));
    }
}
