<?php

namespace App\Services;

use App\Events\LeaveRequestApproved;
use App\Events\LeaveRequestRejected;
use App\Events\LeaveRequestSubmitted;
use App\Models\Employee;
use App\Models\LeaveBalance;
use App\Models\LeaveRequest;
use App\Models\LeaveType;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class LeaveService
{
    public function request(Employee $employee, array $data): LeaveRequest
    {
        return DB::transaction(function () use ($employee, $data) {
            $days = $this->calculateDays(
                $data['start_date'],
                $data['end_date'],
                $data['start_half_day'] ?? false,
                $data['end_half_day'] ?? false
            );

            $balance = $this->getBalance($employee, $data['leave_type_id']);

            if ($balance->available < $days) {
                throw new \Exception('Insufficient leave balance');
            }

            $request = LeaveRequest::create([
                'tenant_id' => $employee->tenant_id,
                'employee_id' => $employee->id,
                'leave_type_id' => $data['leave_type_id'],
                'start_date' => $data['start_date'],
                'end_date' => $data['end_date'],
                'days' => $days,
                'start_half_day' => $data['start_half_day'] ?? false,
                'end_half_day' => $data['end_half_day'] ?? false,
                'reason' => $data['reason'] ?? null,
                'status' => 'pending',
            ]);

            event(new LeaveRequestSubmitted($request));

            return $request;
        });
    }

    public function approve(LeaveRequest $request, int $approverId, ?string $comments = null): void
    {
        DB::transaction(function () use ($request, $approverId, $comments) {
            $request->update([
                'status' => 'approved',
                'approved_by' => $approverId,
                'approved_at' => now(),
                'approver_comments' => $comments,
            ]);

            $balance = $this->getBalance($request->employee, $request->leave_type_id);
            $balance->increment('used', $request->days);

            event(new LeaveRequestApproved($request));
        });
    }

    public function reject(LeaveRequest $request, int $rejectorId, string $reason): void
    {
        $request->update([
            'status' => 'rejected',
            'approved_by' => $rejectorId,
            'approved_at' => now(),
            'rejection_reason' => $reason,
        ]);

        event(new LeaveRequestRejected($request));
    }

    public function cancel(LeaveRequest $request): void
    {
        DB::transaction(function () use ($request) {
            if ($request->status === 'approved') {
                $balance = $this->getBalance($request->employee, $request->leave_type_id);
                $balance->decrement('used', $request->days);
            }

            $request->update(['status' => 'cancelled']);
        });
    }

    public function initializeBalances(Employee $employee): void
    {
        $leaveTypes = LeaveType::where('tenant_id', $employee->tenant_id)
            ->orWhereNull('tenant_id')
            ->where('is_active', true)
            ->get();

        foreach ($leaveTypes as $type) {
            $prorated = $this->calculateProratedEntitlement($employee, $type);

            LeaveBalance::create([
                'tenant_id' => $employee->tenant_id,
                'employee_id' => $employee->id,
                'leave_type_id' => $type->id,
                'year' => now()->year,
                'entitled' => $prorated,
                'used' => 0,
                'carried_forward' => 0,
            ]);
        }
    }

    protected function calculateProratedEntitlement(Employee $employee, LeaveType $type): float
    {
        $hireDate = Carbon::parse($employee->hire_date);
        $yearStart = Carbon::create(now()->year, 1, 1);

        if ($hireDate->year < now()->year) {
            return $type->days_per_year;
        }

        $monthsRemaining = 12 - $hireDate->month + 1;
        return round(($type->days_per_year / 12) * $monthsRemaining, 1);
    }

    protected function calculateDays(string $startDate, string $endDate, bool $startHalf, bool $endHalf): float
    {
        $start = Carbon::parse($startDate);
        $end = Carbon::parse($endDate);
        
        $workingDays = 0;
        $current = $start->copy();

        while ($current <= $end) {
            if (!$current->isWeekend()) {
                $workingDays++;
            }
            $current->addDay();
        }

        if ($startHalf) $workingDays -= 0.5;
        if ($endHalf) $workingDays -= 0.5;

        return max(0, $workingDays);
    }

    protected function getBalance(Employee $employee, int $leaveTypeId): LeaveBalance
    {
        return LeaveBalance::firstOrCreate(
            [
                'employee_id' => $employee->id,
                'leave_type_id' => $leaveTypeId,
                'year' => now()->year,
            ],
            [
                'tenant_id' => $employee->tenant_id,
                'entitled' => 0,
                'used' => 0,
                'carried_forward' => 0,
            ]
        );
    }
}
