<?php

namespace App\Services;

use App\Events\OpportunityLost;
use App\Events\OpportunityStageChanged;
use App\Events\OpportunityWon;
use App\Models\Activity;
use App\Models\Opportunity;
use Illuminate\Support\Facades\DB;

class OpportunityService
{
    public function create(int $tenantId, array $data, int $userId): Opportunity
    {
        return DB::transaction(function () use ($tenantId, $data, $userId) {
            $opportunity = Opportunity::create([
                'tenant_id' => $tenantId,
                'name' => $data['name'],
                'contact_id' => $data['contact_id'],
                'pipeline_id' => $data['pipeline_id'],
                'stage_id' => $data['stage_id'],
                'value' => $data['value'],
                'currency' => $data['currency'] ?? 'BHD',
                'probability' => $data['probability'] ?? 50,
                'expected_close_date' => $data['expected_close_date'] ?? null,
                'description' => $data['description'] ?? null,
                'source' => $data['source'] ?? null,
                'owner_id' => $data['owner_id'] ?? $userId,
                'status' => 'open',
                'stage_changed_at' => now(),
            ]);

            $this->logActivity($opportunity, 'created', "Opportunity created", $userId);

            return $opportunity;
        });
    }

    public function updateStage(Opportunity $opportunity, int $stageId, int $userId): Opportunity
    {
        $oldStage = $opportunity->stage;

        $opportunity->update([
            'stage_id' => $stageId,
            'stage_changed_at' => now(),
        ]);

        $opportunity->refresh();

        $this->logActivity($opportunity, 'stage_changed', 
            "Stage changed from {$oldStage->name} to {$opportunity->stage->name}", $userId);

        event(new OpportunityStageChanged($opportunity, $oldStage, $opportunity->stage));

        return $opportunity;
    }

    public function markWon(Opportunity $opportunity, int $userId, ?array $data = []): Opportunity
    {
        return DB::transaction(function () use ($opportunity, $userId, $data) {
            $opportunity->update([
                'status' => 'won',
                'probability' => 100,
                'closed_at' => now(),
                'closed_by' => $userId,
                'actual_close_date' => $data['close_date'] ?? now(),
            ]);

            $this->logActivity($opportunity, 'won', "Opportunity marked as won", $userId);

            event(new OpportunityWon($opportunity));

            // Update contact status to customer
            $opportunity->contact->update(['lead_status' => 'customer']);

            return $opportunity;
        });
    }

    public function markLost(Opportunity $opportunity, int $userId, string $reason): Opportunity
    {
        $opportunity->update([
            'status' => 'lost',
            'probability' => 0,
            'closed_at' => now(),
            'closed_by' => $userId,
            'lost_reason' => $reason,
        ]);

        $this->logActivity($opportunity, 'lost', "Opportunity marked as lost: {$reason}", $userId);

        event(new OpportunityLost($opportunity));

        return $opportunity;
    }

    public function reopen(Opportunity $opportunity, int $userId): Opportunity
    {
        $opportunity->update([
            'status' => 'open',
            'closed_at' => null,
            'closed_by' => null,
            'lost_reason' => null,
        ]);

        $this->logActivity($opportunity, 'reopened', "Opportunity reopened", $userId);

        return $opportunity;
    }

    protected function logActivity(Opportunity $opportunity, string $type, string $description, int $userId): void
    {
        Activity::create([
            'tenant_id' => $opportunity->tenant_id,
            'activityable_type' => Opportunity::class,
            'activityable_id' => $opportunity->id,
            'type' => $type,
            'description' => $description,
            'user_id' => $userId,
        ]);
    }
}
