<?php

namespace App\Services;

use App\Events\TaskCompleted;
use App\Events\TaskCreated;
use App\Models\Task;
use App\Models\TaskAssignee;
use App\Models\TaskComment;
use App\Models\TaskStatus;
use Illuminate\Support\Facades\DB;

class TaskService
{
    public function create(int $tenantId, array $data, int $userId): Task
    {
        return DB::transaction(function () use ($tenantId, $data, $userId) {
            $statusId = $data['status_id'] ?? $this->getDefaultStatus($data['project_id']);

            $task = Task::create([
                'tenant_id' => $tenantId,
                'project_id' => $data['project_id'],
                'parent_id' => $data['parent_id'] ?? null,
                'status_id' => $statusId,
                'title' => $data['title'],
                'description' => $data['description'] ?? null,
                'type' => $data['type'] ?? 'task',
                'priority' => $data['priority'] ?? 'medium',
                'start_date' => $data['start_date'] ?? null,
                'due_date' => $data['due_date'] ?? null,
                'estimated_hours' => $data['estimated_hours'] ?? null,
                'position' => $this->getNextPosition($data['project_id'], $statusId),
                'created_by' => $userId,
            ]);

            if (!empty($data['assignees'])) {
                foreach ($data['assignees'] as $assigneeId) {
                    TaskAssignee::create([
                        'task_id' => $task->id,
                        'user_id' => $assigneeId,
                    ]);
                }
            }

            event(new TaskCreated($task));

            return $task;
        });
    }

    public function update(Task $task, array $data): Task
    {
        $task->update($data);
        return $task->fresh();
    }

    public function complete(Task $task, int $userId): Task
    {
        $completedStatus = TaskStatus::where('project_id', $task->project_id)
            ->where('is_completed', true)
            ->first();

        $task->update([
            'status_id' => $completedStatus?->id ?? $task->status_id,
            'completed_at' => now(),
            'completed_by' => $userId,
        ]);

        event(new TaskCompleted($task));

        return $task;
    }

    public function reopen(Task $task): Task
    {
        $defaultStatus = $this->getDefaultStatus($task->project_id);

        $task->update([
            'status_id' => $defaultStatus,
            'completed_at' => null,
            'completed_by' => null,
        ]);

        return $task;
    }

    public function addComment(Task $task, string $content, int $userId, ?int $parentId = null): TaskComment
    {
        return TaskComment::create([
            'tenant_id' => $task->tenant_id,
            'task_id' => $task->id,
            'parent_id' => $parentId,
            'user_id' => $userId,
            'content' => $content,
        ]);
    }

    public function updateAssignees(Task $task, array $userIds): void
    {
        $task->assignees()->delete();

        foreach ($userIds as $userId) {
            TaskAssignee::create([
                'task_id' => $task->id,
                'user_id' => $userId,
            ]);
        }
    }

    public function moveToStatus(Task $task, int $statusId, ?int $position = null): Task
    {
        $task->update([
            'status_id' => $statusId,
            'position' => $position ?? $this->getNextPosition($task->project_id, $statusId),
        ]);

        return $task;
    }

    protected function getDefaultStatus(int $projectId): int
    {
        return TaskStatus::where('project_id', $projectId)
            ->where('is_default', true)
            ->value('id') ?? TaskStatus::where('project_id', $projectId)->min('id');
    }

    protected function getNextPosition(int $projectId, int $statusId): int
    {
        return Task::where('project_id', $projectId)
            ->where('status_id', $statusId)
            ->max('position') + 1;
    }
}
