<?php

namespace App\Services;

use App\Events\WorkflowStepCompleted;
use App\Jobs\ProcessWorkflowStep;
use App\Models\Contact;
use App\Models\Workflow;
use App\Models\WorkflowEnrollment;
use App\Models\WorkflowStep;
use Carbon\Carbon;

class WorkflowService
{
    public function create(int $tenantId, array $data): Workflow
    {
        $workflow = Workflow::create([
            'tenant_id' => $tenantId,
            'name' => $data['name'],
            'description' => $data['description'] ?? null,
            'trigger_type' => $data['trigger_type'],
            'trigger_conditions' => $data['trigger_conditions'] ?? [],
            'is_active' => false,
        ]);

        foreach ($data['steps'] ?? [] as $index => $stepData) {
            WorkflowStep::create([
                'workflow_id' => $workflow->id,
                'order' => $index + 1,
                'type' => $stepData['type'],
                'config' => $stepData['config'],
                'delay_minutes' => $stepData['delay_minutes'] ?? 0,
            ]);
        }

        return $workflow->load('steps');
    }

    public function enroll(Workflow $workflow, Contact $contact): WorkflowEnrollment
    {
        $enrollment = WorkflowEnrollment::create([
            'tenant_id' => $workflow->tenant_id,
            'workflow_id' => $workflow->id,
            'contact_id' => $contact->id,
            'status' => 'active',
            'enrolled_at' => now(),
            'current_step' => 1,
        ]);

        $this->scheduleNextStep($enrollment);

        return $enrollment;
    }

    public function processStep(WorkflowEnrollment $enrollment): void
    {
        $step = $enrollment->workflow->steps()
            ->where('order', $enrollment->current_step)
            ->first();

        if (!$step) {
            $this->completeEnrollment($enrollment);
            return;
        }

        $this->executeStep($enrollment, $step);

        $enrollment->update([
            'current_step' => $enrollment->current_step + 1,
            'last_step_at' => now(),
        ]);

        event(new WorkflowStepCompleted($enrollment, $step));

        if ($enrollment->workflow->steps()->where('order', '>', $step->order)->exists()) {
            $this->scheduleNextStep($enrollment);
        } else {
            $this->completeEnrollment($enrollment);
        }
    }

    public function unenroll(WorkflowEnrollment $enrollment, string $reason = 'manual'): void
    {
        $enrollment->update([
            'status' => 'unenrolled',
            'completed_at' => now(),
            'unenroll_reason' => $reason,
        ]);
    }

    public function triggerByEvent(int $tenantId, string $eventType, Contact $contact, array $data = []): void
    {
        $workflows = Workflow::where('tenant_id', $tenantId)
            ->where('trigger_type', $eventType)
            ->where('is_active', true)
            ->get();

        foreach ($workflows as $workflow) {
            if ($this->checkConditions($workflow, $contact, $data)) {
                if (!$this->isAlreadyEnrolled($workflow, $contact)) {
                    $this->enroll($workflow, $contact);
                }
            }
        }
    }

    protected function executeStep(WorkflowEnrollment $enrollment, WorkflowStep $step): void
    {
        $contact = $enrollment->contact;
        $config = $step->config;

        match ($step->type) {
            'send_email' => $this->sendEmail($contact, $config),
            'send_sms' => $this->sendSms($contact, $config),
            'add_tag' => $contact->tags()->syncWithoutDetaching([$config['tag_id']]),
            'remove_tag' => $contact->tags()->detach($config['tag_id']),
            'update_field' => $contact->update([$config['field'] => $config['value']]),
            'webhook' => $this->sendWebhook($contact, $config),
            'wait' => null,
            default => null,
        };
    }

    protected function scheduleNextStep(WorkflowEnrollment $enrollment): void
    {
        $nextStep = $enrollment->workflow->steps()
            ->where('order', $enrollment->current_step)
            ->first();

        if (!$nextStep) return;

        $delay = $nextStep->delay_minutes;
        $runAt = now()->addMinutes($delay);

        ProcessWorkflowStep::dispatch($enrollment)->delay($runAt);
    }

    protected function completeEnrollment(WorkflowEnrollment $enrollment): void
    {
        $enrollment->update([
            'status' => 'completed',
            'completed_at' => now(),
        ]);
    }

    protected function checkConditions(Workflow $workflow, Contact $contact, array $data): bool
    {
        $conditions = $workflow->trigger_conditions;
        if (empty($conditions)) return true;

        foreach ($conditions as $condition) {
            $field = $condition['field'];
            $operator = $condition['operator'];
            $value = $condition['value'];
            $contactValue = $contact->$field ?? $data[$field] ?? null;

            $match = match ($operator) {
                'equals' => $contactValue == $value,
                'not_equals' => $contactValue != $value,
                'contains' => str_contains($contactValue, $value),
                'starts_with' => str_starts_with($contactValue, $value),
                default => false,
            };

            if (!$match) return false;
        }

        return true;
    }

    protected function isAlreadyEnrolled(Workflow $workflow, Contact $contact): bool
    {
        return WorkflowEnrollment::where('workflow_id', $workflow->id)
            ->where('contact_id', $contact->id)
            ->whereIn('status', ['active', 'completed'])
            ->exists();
    }

    protected function sendEmail(Contact $contact, array $config): void
    {
        // Implementation would use email service
    }

    protected function sendSms(Contact $contact, array $config): void
    {
        // Implementation would use SMS service
    }

    protected function sendWebhook(Contact $contact, array $config): void
    {
        // Implementation would send HTTP request
    }
}
