#!/bin/bash
# Business Platform Deployment Script

set -e

echo "🚀 Starting deployment..."

# Colors
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

# Check if .env exists
if [ ! -f .env ]; then
    echo -e "${YELLOW}Creating .env file...${NC}"
    cp .env.example .env
fi

# Install Composer dependencies
echo -e "${GREEN}Installing PHP dependencies...${NC}"
composer install --no-dev --optimize-autoloader

# Install NPM dependencies and build
echo -e "${GREEN}Installing NPM dependencies...${NC}"
npm ci
npm run build

# Generate application key
echo -e "${GREEN}Generating application key...${NC}"
php artisan key:generate --force

# Run migrations
echo -e "${GREEN}Running database migrations...${NC}"
php artisan migrate --force

# Seed database (optional)
read -p "Run database seeders? (y/n) " -n 1 -r
echo
if [[ $REPLY =~ ^[Yy]$ ]]; then
    php artisan db:seed --force
fi

# Clear and cache
echo -e "${GREEN}Optimizing application...${NC}"
php artisan config:cache
php artisan route:cache
php artisan view:cache
php artisan icons:cache

# Set permissions
echo -e "${GREEN}Setting permissions...${NC}"
chmod -R 775 storage bootstrap/cache
chown -R www-data:www-data storage bootstrap/cache

# Create storage link
php artisan storage:link

echo -e "${GREEN}✅ Deployment complete!${NC}"
echo ""
echo "Next steps:"
echo "1. Configure your web server (Nginx/Apache)"
echo "2. Set up SSL certificate"
echo "3. Configure queue worker: php artisan queue:work"
echo "4. Set up cron for scheduler: * * * * * php artisan schedule:run"
