# 🤖 BOT 2: CRM MODULE

## 📋 YOUR RESPONSIBILITY
Build complete CRM functionality: Contacts, Companies, Pipelines, Opportunities, Activities, Smart Lists.

---

## 📁 FILES TO CREATE

### 1. MODELS (12 files)
```
app/Models/
├── Contact.php ✅ EXISTS - ENHANCE
├── Pipeline.php ✅ EXISTS - ENHANCE
├── PipelineStage.php ✅ EXISTS - ENHANCE
├── Opportunity.php ✅ EXISTS - ENHANCE
├── Activity.php - CREATE
├── Tag.php ✅ EXISTS - ENHANCE
├── SmartList.php - CREATE
├── CustomFieldDefinition.php - CREATE
├── Product.php - CREATE
├── OpportunityProduct.php - CREATE
├── ContactNote.php - CREATE
├── ContactTask.php - CREATE
```

### 2. CONTROLLERS (8 files)
```
app/Http/Controllers/Api/
├── ContactController.php ✅ EXISTS - ENHANCE
├── PipelineController.php ✅ EXISTS - ENHANCE
├── OpportunityController.php ✅ EXISTS - ENHANCE
├── ActivityController.php - CREATE
├── TagController.php - CREATE
├── SmartListController.php - CREATE
├── CustomFieldController.php - CREATE
├── ProductController.php - CREATE
```

### 3. SERVICES (6 files)
```
app/Services/
├── ContactService.php - CREATE
├── PipelineService.php - CREATE
├── OpportunityService.php - CREATE
├── ActivityService.php - CREATE
├── ImportExportService.php - CREATE
├── LeadScoringService.php - CREATE
```

### 4. FORM REQUESTS (12 files)
```
app/Http/Requests/
├── Contact/StoreContactRequest.php
├── Contact/UpdateContactRequest.php
├── Contact/ImportContactRequest.php
├── Contact/MergeContactRequest.php
├── Pipeline/StorePipelineRequest.php
├── Pipeline/UpdatePipelineRequest.php
├── Opportunity/StoreOpportunityRequest.php
├── Opportunity/UpdateOpportunityRequest.php
├── Activity/StoreActivityRequest.php
├── Tag/StoreTagRequest.php
├── SmartList/StoreSmartListRequest.php
├── Product/StoreProductRequest.php
```

### 5. ROUTES (1 file)
```
routes/api/crm.php - CREATE
```

### 6. JOBS (5 files)
```
app/Jobs/
├── ImportContacts.php
├── ExportContacts.php
├── RecalculateLeadScores.php
├── SyncContactsToMailchimp.php
├── UpdateOpportunityRotting.php
```

### 7. EVENTS & LISTENERS (6 files)
```
app/Events/
├── ContactCreated.php
├── ContactUpdated.php
├── OpportunityStageChanged.php
├── OpportunityWon.php
├── OpportunityLost.php

app/Listeners/
├── LogContactActivity.php
├── NotifyOnOpportunityWon.php
├── UpdateContactLifetimeValue.php
├── TriggerWorkflowOnEvent.php
```

### 8. NOTIFICATIONS (3 files)
```
app/Notifications/
├── OpportunityAssignedNotification.php
├── OpportunityWonNotification.php
├── TaskReminderNotification.php
```

### 9. POLICIES (4 files)
```
app/Policies/
├── ContactPolicy.php
├── OpportunityPolicy.php
├── PipelinePolicy.php
├── ActivityPolicy.php
```

### 10. SEEDERS (3 files)
```
database/seeders/
├── ContactSeeder.php
├── PipelineSeeder.php
├── ProductSeeder.php
```

### 11. TESTS (6 files)
```
tests/Feature/
├── ContactTest.php
├── PipelineTest.php
├── OpportunityTest.php
├── ActivityTest.php
├── SmartListTest.php
├── ContactImportTest.php
```

---

## 📝 DETAILED SPECIFICATIONS

### ContactController.php (ENHANCE)
```php
<?php
namespace App\Http\Controllers\Api;

class ContactController extends ApiController
{
    // GET /api/contacts - List with filters
    public function index(Request $request);
    
    // POST /api/contacts - Create contact
    public function store(StoreContactRequest $request);
    
    // GET /api/contacts/{contact} - Show contact
    public function show(Contact $contact);
    
    // PUT /api/contacts/{contact} - Update
    public function update(UpdateContactRequest $request, Contact $contact);
    
    // DELETE /api/contacts/{contact} - Soft delete
    public function destroy(Contact $contact);
    
    // POST /api/contacts/import - Bulk import CSV/Excel
    public function import(ImportContactRequest $request);
    
    // GET /api/contacts/export - Export to CSV
    public function export(Request $request);
    
    // POST /api/contacts/merge - Merge duplicates
    public function merge(MergeContactRequest $request);
    
    // GET /api/contacts/{contact}/timeline - Activity timeline
    public function timeline(Contact $contact);
    
    // GET /api/contacts/{contact}/opportunities - Contact opportunities
    public function opportunities(Contact $contact);
    
    // GET /api/contacts/{contact}/invoices - Contact invoices
    public function invoices(Contact $contact);
    
    // POST /api/contacts/{contact}/tags - Add tags
    public function addTags(Request $request, Contact $contact);
    
    // DELETE /api/contacts/{contact}/tags/{tag} - Remove tag
    public function removeTag(Contact $contact, Tag $tag);
    
    // POST /api/contacts/{contact}/convert - Convert lead to customer
    public function convert(Contact $contact);
    
    // GET /api/contacts/duplicates - Find duplicates
    public function duplicates();
}
```

### ActivityController.php
```php
<?php
namespace App\Http\Controllers\Api;

class ActivityController extends ApiController
{
    // GET /api/activities - List activities
    public function index(Request $request);
    
    // POST /api/activities - Log activity (call, email, meeting, note)
    public function store(StoreActivityRequest $request);
    
    // GET /api/activities/{activity} - Show activity
    public function show(Activity $activity);
    
    // PUT /api/activities/{activity} - Update
    public function update(Request $request, Activity $activity);
    
    // DELETE /api/activities/{activity} - Delete
    public function destroy(Activity $activity);
    
    // POST /api/activities/{activity}/complete - Mark complete
    public function complete(Activity $activity);
    
    // GET /api/activities/upcoming - Upcoming scheduled activities
    public function upcoming();
    
    // GET /api/activities/overdue - Overdue activities
    public function overdue();
}
```

### SmartListController.php
```php
<?php
namespace App\Http\Controllers\Api;

class SmartListController extends ApiController
{
    // GET /api/smart-lists - List saved filters
    public function index();
    
    // POST /api/smart-lists - Create smart list
    public function store(StoreSmartListRequest $request);
    
    // GET /api/smart-lists/{smartList} - Get smart list
    public function show(SmartList $smartList);
    
    // PUT /api/smart-lists/{smartList} - Update
    public function update(Request $request, SmartList $smartList);
    
    // DELETE /api/smart-lists/{smartList} - Delete
    public function destroy(SmartList $smartList);
    
    // GET /api/smart-lists/{smartList}/contacts - Get matching contacts
    public function contacts(SmartList $smartList);
    
    // GET /api/smart-lists/{smartList}/count - Get count
    public function count(SmartList $smartList);
}
```

---

## 🗄️ DATABASE TABLES
- contacts
- contact_tag
- tags
- pipelines
- pipeline_stages
- opportunities
- opportunity_products
- activities
- products
- smart_lists
- custom_field_definitions

---

## 🔗 ROUTES TO DEFINE

```php
// routes/api/crm.php

Route::middleware('auth:sanctum')->group(function () {
    // Contacts
    Route::apiResource('contacts', ContactController::class);
    Route::post('contacts/import', [ContactController::class, 'import']);
    Route::get('contacts/export', [ContactController::class, 'export']);
    Route::post('contacts/merge', [ContactController::class, 'merge']);
    Route::get('contacts/duplicates', [ContactController::class, 'duplicates']);
    Route::get('contacts/{contact}/timeline', [ContactController::class, 'timeline']);
    Route::get('contacts/{contact}/opportunities', [ContactController::class, 'opportunities']);
    Route::get('contacts/{contact}/invoices', [ContactController::class, 'invoices']);
    Route::post('contacts/{contact}/tags', [ContactController::class, 'addTags']);
    Route::delete('contacts/{contact}/tags/{tag}', [ContactController::class, 'removeTag']);
    Route::post('contacts/{contact}/convert', [ContactController::class, 'convert']);
    
    // Pipelines
    Route::apiResource('pipelines', PipelineController::class);
    Route::post('pipelines/{pipeline}/stages/reorder', [PipelineController::class, 'reorderStages']);
    Route::get('pipelines/{pipeline}/kanban', [OpportunityController::class, 'kanban']);
    
    // Pipeline Stages
    Route::apiResource('pipelines.stages', PipelineStageController::class)->shallow();
    
    // Opportunities
    Route::apiResource('opportunities', OpportunityController::class);
    Route::post('opportunities/{opportunity}/move', [OpportunityController::class, 'moveStage']);
    Route::post('opportunities/{opportunity}/won', [OpportunityController::class, 'markWon']);
    Route::post('opportunities/{opportunity}/lost', [OpportunityController::class, 'markLost']);
    Route::get('opportunities/{opportunity}/products', [OpportunityController::class, 'products']);
    Route::post('opportunities/{opportunity}/products', [OpportunityController::class, 'addProduct']);
    
    // Activities
    Route::apiResource('activities', ActivityController::class);
    Route::post('activities/{activity}/complete', [ActivityController::class, 'complete']);
    Route::get('activities/upcoming', [ActivityController::class, 'upcoming']);
    Route::get('activities/overdue', [ActivityController::class, 'overdue']);
    
    // Tags
    Route::apiResource('tags', TagController::class);
    
    // Smart Lists
    Route::apiResource('smart-lists', SmartListController::class);
    Route::get('smart-lists/{smartList}/contacts', [SmartListController::class, 'contacts']);
    Route::get('smart-lists/{smartList}/count', [SmartListController::class, 'count']);
    
    // Products
    Route::apiResource('products', ProductController::class);
    
    // Custom Fields
    Route::apiResource('custom-fields', CustomFieldController::class);
    
    // CRM Dashboard
    Route::get('crm/dashboard', [CrmDashboardController::class, 'index']);
    Route::get('crm/reports/pipeline', [CrmReportController::class, 'pipeline']);
    Route::get('crm/reports/activities', [CrmReportController::class, 'activities']);
    Route::get('crm/reports/conversion', [CrmReportController::class, 'conversion']);
});
```

---

## 🇧🇭 BAHRAIN-SPECIFIC FEATURES

1. **Contact Fields**:
   - CR Number (Commercial Registration)
   - Nationality
   - Ownership Type (100% foreign, 51% Bahraini, GCC, mixed)
   - Business Sector

2. **Opportunity Fields**:
   - Package (Standard BHD 1,340 / Gold BHD 1,700 / Premium BHD 2,150)
   - Legal Form (WLL, SPC, Branch)
   - Business Activity

3. **Currency**: Default BHD with multi-currency support

4. **WhatsApp Integration**: WhatsApp number field, click-to-chat (wa.link/setupinbahrain)

---

## ✅ COMPLETION CHECKLIST

- [ ] All models created with relationships
- [ ] All controllers with full CRUD + custom actions
- [ ] All services with business logic
- [ ] Contact import/export working
- [ ] Kanban board for opportunities
- [ ] Smart lists with dynamic filtering
- [ ] Activity logging
- [ ] Lead scoring
- [ ] All routes defined
- [ ] All tests passing

---

## 📤 OUTPUT LOCATION

Save all files to: `/home/claude/business-platform/`

When complete, update: `/home/claude/business-platform/docs/PROGRESS_BOT_2.md`
