# 🤖 BOT 3: ACCOUNTING - INVOICING MODULE

## 📋 YOUR RESPONSIBILITY
Build invoicing, estimates/quotes, payments received, and recurring invoices.

---

## 📁 FILES TO CREATE

### 1. MODELS (10 files)
```
app/Models/
├── Invoice.php ✅ EXISTS - ENHANCE
├── InvoiceItem.php - CREATE
├── Estimate.php - CREATE
├── EstimateItem.php - CREATE
├── Payment.php - CREATE
├── RecurringInvoice.php - CREATE
├── TaxRate.php - CREATE
├── Currency.php - CREATE
├── PaymentTerm.php - CREATE
├── InvoiceTemplate.php - CREATE
```

### 2. CONTROLLERS (6 files)
```
app/Http/Controllers/Api/
├── InvoiceController.php ✅ EXISTS - ENHANCE
├── EstimateController.php - CREATE
├── PaymentController.php - CREATE
├── RecurringInvoiceController.php - CREATE
├── TaxRateController.php - CREATE
├── CurrencyController.php - CREATE
```

### 3. SERVICES (5 files)
```
app/Services/
├── InvoiceService.php - CREATE
├── EstimateService.php - CREATE
├── PaymentService.php - CREATE
├── RecurringInvoiceService.php - CREATE
├── PdfGeneratorService.php - CREATE
```

### 4. FORM REQUESTS (10 files)
```
app/Http/Requests/
├── Invoice/StoreInvoiceRequest.php
├── Invoice/UpdateInvoiceRequest.php
├── Invoice/SendInvoiceRequest.php
├── Estimate/StoreEstimateRequest.php
├── Estimate/UpdateEstimateRequest.php
├── Payment/StorePaymentRequest.php
├── Payment/RefundPaymentRequest.php
├── RecurringInvoice/StoreRecurringInvoiceRequest.php
├── TaxRate/StoreTaxRateRequest.php
├── Currency/StoreCurrencyRequest.php
```

### 5. ROUTES (1 file)
```
routes/api/invoicing.php - CREATE
```

### 6. JOBS (4 files)
```
app/Jobs/
├── SendInvoiceEmail.php
├── GenerateRecurringInvoices.php
├── SendPaymentReminder.php
├── MarkOverdueInvoices.php
```

### 7. EVENTS & LISTENERS (6 files)
```
app/Events/
├── InvoiceCreated.php
├── InvoiceSent.php
├── InvoicePaid.php
├── PaymentReceived.php

app/Listeners/
├── CreateJournalEntryOnInvoice.php
├── CreateJournalEntryOnPayment.php
├── UpdateContactLifetimeValue.php
├── NotifyOnPaymentReceived.php
```

### 8. NOTIFICATIONS (4 files)
```
app/Notifications/
├── InvoiceSentNotification.php
├── PaymentReceivedNotification.php
├── PaymentReminderNotification.php
├── InvoiceOverdueNotification.php
```

### 9. VIEWS (4 files)
```
resources/views/invoices/
├── pdf.blade.php
├── email.blade.php

resources/views/estimates/
├── pdf.blade.php
├── email.blade.php
```

### 10. SEEDERS (2 files)
```
database/seeders/
├── TaxRateSeeder.php
├── CurrencySeeder.php
```

### 11. TESTS (5 files)
```
tests/Feature/
├── InvoiceTest.php
├── EstimateTest.php
├── PaymentTest.php
├── RecurringInvoiceTest.php
├── InvoicePdfTest.php
```

---

## 📝 DETAILED SPECIFICATIONS

### InvoiceController.php (ENHANCE)
```php
<?php
namespace App\Http\Controllers\Api;

class InvoiceController extends ApiController
{
    // GET /api/invoices - List invoices with filters
    public function index(Request $request);
    
    // POST /api/invoices - Create invoice
    public function store(StoreInvoiceRequest $request);
    
    // GET /api/invoices/{invoice} - Show invoice
    public function show(Invoice $invoice);
    
    // PUT /api/invoices/{invoice} - Update draft invoice
    public function update(UpdateInvoiceRequest $request, Invoice $invoice);
    
    // DELETE /api/invoices/{invoice} - Delete draft invoice
    public function destroy(Invoice $invoice);
    
    // POST /api/invoices/{invoice}/send - Send invoice via email
    public function send(SendInvoiceRequest $request, Invoice $invoice);
    
    // POST /api/invoices/{invoice}/resend - Resend invoice
    public function resend(Invoice $invoice);
    
    // GET /api/invoices/{invoice}/pdf - Download PDF
    public function pdf(Invoice $invoice);
    
    // GET /api/invoices/{invoice}/preview - Preview PDF in browser
    public function preview(Invoice $invoice);
    
    // POST /api/invoices/{invoice}/payment - Record payment
    public function recordPayment(StorePaymentRequest $request, Invoice $invoice);
    
    // POST /api/invoices/{invoice}/void - Void invoice
    public function void(Invoice $invoice);
    
    // POST /api/invoices/{invoice}/duplicate - Duplicate invoice
    public function duplicate(Invoice $invoice);
    
    // POST /api/invoices/{invoice}/credit-note - Create credit note
    public function createCreditNote(Invoice $invoice);
    
    // GET /api/invoices/summary - Invoice summary/stats
    public function summary(Request $request);
    
    // GET /api/invoices/aging - Aging report
    public function aging();
    
    // GET /api/invoices/overdue - Overdue invoices
    public function overdue();
}
```

### EstimateController.php
```php
<?php
namespace App\Http\Controllers\Api;

class EstimateController extends ApiController
{
    // GET /api/estimates - List estimates
    public function index(Request $request);
    
    // POST /api/estimates - Create estimate
    public function store(StoreEstimateRequest $request);
    
    // GET /api/estimates/{estimate} - Show
    public function show(Estimate $estimate);
    
    // PUT /api/estimates/{estimate} - Update
    public function update(UpdateEstimateRequest $request, Estimate $estimate);
    
    // DELETE /api/estimates/{estimate} - Delete
    public function destroy(Estimate $estimate);
    
    // POST /api/estimates/{estimate}/send - Send to client
    public function send(Estimate $estimate);
    
    // GET /api/estimates/{estimate}/pdf - Download PDF
    public function pdf(Estimate $estimate);
    
    // POST /api/estimates/{estimate}/accept - Mark accepted
    public function accept(Estimate $estimate);
    
    // POST /api/estimates/{estimate}/decline - Mark declined
    public function decline(Request $request, Estimate $estimate);
    
    // POST /api/estimates/{estimate}/convert - Convert to invoice
    public function convertToInvoice(Estimate $estimate);
    
    // POST /api/estimates/{estimate}/duplicate - Duplicate
    public function duplicate(Estimate $estimate);
}
```

### RecurringInvoiceController.php
```php
<?php
namespace App\Http\Controllers\Api;

class RecurringInvoiceController extends ApiController
{
    // GET /api/recurring-invoices - List
    public function index(Request $request);
    
    // POST /api/recurring-invoices - Create
    public function store(StoreRecurringInvoiceRequest $request);
    
    // GET /api/recurring-invoices/{recurring} - Show
    public function show(RecurringInvoice $recurring);
    
    // PUT /api/recurring-invoices/{recurring} - Update
    public function update(Request $request, RecurringInvoice $recurring);
    
    // DELETE /api/recurring-invoices/{recurring} - Delete
    public function destroy(RecurringInvoice $recurring);
    
    // POST /api/recurring-invoices/{recurring}/pause - Pause
    public function pause(RecurringInvoice $recurring);
    
    // POST /api/recurring-invoices/{recurring}/resume - Resume
    public function resume(RecurringInvoice $recurring);
    
    // POST /api/recurring-invoices/{recurring}/generate - Generate now
    public function generateNow(RecurringInvoice $recurring);
    
    // GET /api/recurring-invoices/{recurring}/invoices - List generated
    public function invoices(RecurringInvoice $recurring);
}
```

---

## 🗄️ DATABASE TABLES
- invoices
- invoice_items
- estimates
- estimate_items
- payments
- recurring_invoices
- tax_rates
- currencies

---

## 🔗 ROUTES TO DEFINE

```php
// routes/api/invoicing.php

Route::middleware('auth:sanctum')->group(function () {
    // Invoices
    Route::apiResource('invoices', InvoiceController::class);
    Route::post('invoices/{invoice}/send', [InvoiceController::class, 'send']);
    Route::post('invoices/{invoice}/resend', [InvoiceController::class, 'resend']);
    Route::get('invoices/{invoice}/pdf', [InvoiceController::class, 'pdf']);
    Route::get('invoices/{invoice}/preview', [InvoiceController::class, 'preview']);
    Route::post('invoices/{invoice}/payment', [InvoiceController::class, 'recordPayment']);
    Route::post('invoices/{invoice}/void', [InvoiceController::class, 'void']);
    Route::post('invoices/{invoice}/duplicate', [InvoiceController::class, 'duplicate']);
    Route::post('invoices/{invoice}/credit-note', [InvoiceController::class, 'createCreditNote']);
    Route::get('invoices-summary', [InvoiceController::class, 'summary']);
    Route::get('invoices-aging', [InvoiceController::class, 'aging']);
    Route::get('invoices-overdue', [InvoiceController::class, 'overdue']);
    
    // Estimates
    Route::apiResource('estimates', EstimateController::class);
    Route::post('estimates/{estimate}/send', [EstimateController::class, 'send']);
    Route::get('estimates/{estimate}/pdf', [EstimateController::class, 'pdf']);
    Route::post('estimates/{estimate}/accept', [EstimateController::class, 'accept']);
    Route::post('estimates/{estimate}/decline', [EstimateController::class, 'decline']);
    Route::post('estimates/{estimate}/convert', [EstimateController::class, 'convertToInvoice']);
    Route::post('estimates/{estimate}/duplicate', [EstimateController::class, 'duplicate']);
    
    // Payments
    Route::apiResource('payments', PaymentController::class);
    Route::post('payments/{payment}/refund', [PaymentController::class, 'refund']);
    
    // Recurring Invoices
    Route::apiResource('recurring-invoices', RecurringInvoiceController::class);
    Route::post('recurring-invoices/{recurring}/pause', [RecurringInvoiceController::class, 'pause']);
    Route::post('recurring-invoices/{recurring}/resume', [RecurringInvoiceController::class, 'resume']);
    Route::post('recurring-invoices/{recurring}/generate', [RecurringInvoiceController::class, 'generateNow']);
    Route::get('recurring-invoices/{recurring}/invoices', [RecurringInvoiceController::class, 'invoices']);
    
    // Tax Rates
    Route::apiResource('tax-rates', TaxRateController::class);
    
    // Currencies
    Route::apiResource('currencies', CurrencyController::class);
    Route::post('currencies/{currency}/update-rate', [CurrencyController::class, 'updateRate']);
});

// Public invoice view (for clients)
Route::get('invoices/{invoice}/view/{token}', [PublicInvoiceController::class, 'show']);
Route::post('invoices/{invoice}/pay/{token}', [PublicInvoiceController::class, 'pay']);
```

---

## 🇧🇭 BAHRAIN-SPECIFIC FEATURES

1. **Tax**: 10% VAT (Value Added Tax)
2. **Currency**: BHD (3 decimal places)
3. **Invoice Format**: Include CR number, VAT number
4. **Multi-currency**: Support BHD, USD, SAR, AED, EUR, GBP
5. **Payment Methods**: Bank transfer, credit card, cash

---

## 🎨 PDF TEMPLATE (pdf.blade.php)
```html
<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <title>Invoice {{ $invoice->invoice_number }}</title>
    <style>
        body { font-family: Arial, sans-serif; font-size: 12px; }
        .header { display: flex; justify-content: space-between; }
        .logo { max-width: 150px; }
        .invoice-info { text-align: right; }
        .addresses { display: flex; justify-content: space-between; margin: 30px 0; }
        table { width: 100%; border-collapse: collapse; }
        th, td { padding: 10px; border: 1px solid #ddd; }
        th { background: #f5f5f5; }
        .totals { width: 300px; margin-left: auto; }
        .totals td:last-child { text-align: right; }
        .footer { margin-top: 50px; font-size: 10px; color: #666; }
    </style>
</head>
<body>
    <div class="header">
        <img src="{{ $tenant->logo }}" class="logo">
        <div class="invoice-info">
            <h1>INVOICE</h1>
            <p><strong>{{ $invoice->invoice_number }}</strong></p>
            <p>Date: {{ $invoice->invoice_date->format('d/m/Y') }}</p>
            <p>Due: {{ $invoice->due_date->format('d/m/Y') }}</p>
        </div>
    </div>
    
    <div class="addresses">
        <div class="from">
            <strong>From:</strong><br>
            {{ $tenant->name }}<br>
            {{ $tenant->address }}<br>
            CR: {{ $tenant->tax_number }}<br>
            VAT: {{ $tenant->vat_number }}
        </div>
        <div class="to">
            <strong>Bill To:</strong><br>
            {{ $invoice->contact->company_name ?? $invoice->contact->full_name }}<br>
            {{ $invoice->billing_address['address'] ?? '' }}<br>
            CR: {{ $invoice->contact->cr_number ?? 'N/A' }}
        </div>
    </div>
    
    <table>
        <thead>
            <tr>
                <th>#</th>
                <th>Description</th>
                <th>Qty</th>
                <th>Unit Price</th>
                <th>Tax</th>
                <th>Total</th>
            </tr>
        </thead>
        <tbody>
            @foreach($invoice->items as $index => $item)
            <tr>
                <td>{{ $index + 1 }}</td>
                <td>{{ $item->name }}<br><small>{{ $item->description }}</small></td>
                <td>{{ $item->quantity }}</td>
                <td>{{ $invoice->currency }} {{ number_format($item->unit_price, 3) }}</td>
                <td>{{ number_format($item->tax_amount, 3) }}</td>
                <td>{{ $invoice->currency }} {{ number_format($item->total, 3) }}</td>
            </tr>
            @endforeach
        </tbody>
    </table>
    
    <table class="totals">
        <tr><td>Subtotal:</td><td>{{ $invoice->currency }} {{ number_format($invoice->subtotal, 3) }}</td></tr>
        <tr><td>VAT (10%):</td><td>{{ $invoice->currency }} {{ number_format($invoice->tax_amount, 3) }}</td></tr>
        <tr><td><strong>Total:</strong></td><td><strong>{{ $invoice->currency }} {{ number_format($invoice->total, 3) }}</strong></td></tr>
        @if($invoice->amount_paid > 0)
        <tr><td>Paid:</td><td>{{ $invoice->currency }} {{ number_format($invoice->amount_paid, 3) }}</td></tr>
        <tr><td><strong>Balance Due:</strong></td><td><strong>{{ $invoice->currency }} {{ number_format($invoice->amount_due, 3) }}</strong></td></tr>
        @endif
    </table>
    
    @if($invoice->notes)
    <div class="notes">
        <strong>Notes:</strong><br>
        {{ $invoice->notes }}
    </div>
    @endif
    
    <div class="footer">
        {{ $invoice->terms ?? $tenant->default_invoice_terms }}
    </div>
</body>
</html>
```

---

## ✅ COMPLETION CHECKLIST

- [ ] All models with relationships
- [ ] Full invoice CRUD + send/pay/void
- [ ] Estimate CRUD + convert to invoice
- [ ] Recurring invoice generation
- [ ] PDF generation working
- [ ] Email templates
- [ ] Payment recording
- [ ] Tax calculations (10% VAT)
- [ ] Multi-currency support
- [ ] All routes defined
- [ ] All tests passing

---

## 📤 OUTPUT LOCATION

Save all files to: `/home/claude/business-platform/`

When complete, update: `/home/claude/business-platform/docs/PROGRESS_BOT_3.md`
