# 🤖 BOT 6: HR / PAYROLL MODULE

## 📋 YOUR RESPONSIBILITY
Build employee management, payroll processing, leave management, attendance, and HR reporting.

---

## 📁 FILES TO CREATE

### 1. MODELS (14 files)
```
app/Models/
├── Employee.php ✅ EXISTS - ENHANCE
├── PayPeriod.php - CREATE
├── PayrollRun.php - CREATE
├── PayrollItem.php - CREATE
├── SalaryComponent.php - CREATE
├── EmployeeSalaryComponent.php - CREATE
├── LeaveType.php - CREATE
├── LeaveBalance.php - CREATE
├── LeaveRequest.php - CREATE
├── Attendance.php - CREATE
├── EmployeeLoan.php - CREATE
├── Holiday.php - CREATE
├── Department.php - CREATE (alias for Team)
├── EmployeeDocument.php - CREATE
```

### 2. CONTROLLERS (8 files)
```
app/Http/Controllers/Api/
├── EmployeeController.php - CREATE
├── PayrollController.php - CREATE
├── LeaveController.php - CREATE
├── AttendanceController.php - CREATE
├── EmployeeLoanController.php - CREATE
├── HolidayController.php - CREATE
├── SalaryComponentController.php - CREATE
├── HRReportController.php - CREATE
```

### 3. SERVICES (5 files)
```
app/Services/
├── EmployeeService.php
├── PayrollService.php
├── LeaveService.php
├── AttendanceService.php
├── PayslipPdfService.php
```

### 4. FORM REQUESTS (12 files)
```
app/Http/Requests/
├── Employee/StoreEmployeeRequest.php
├── Employee/UpdateEmployeeRequest.php
├── Payroll/CreatePayrollRunRequest.php
├── Payroll/ApprovePayrollRequest.php
├── Leave/StoreLeaveTypeRequest.php
├── Leave/StoreLeaveRequestRequest.php
├── Leave/ApproveLeaveRequest.php
├── Attendance/StoreAttendanceRequest.php
├── Attendance/BulkAttendanceRequest.php
├── Loan/StoreLoanRequest.php
├── Holiday/StoreHolidayRequest.php
├── SalaryComponent/StoreComponentRequest.php
```

### 5. ROUTES (1 file)
```
routes/api/hr.php - CREATE
```

### 6. JOBS (4 files)
```
app/Jobs/
├── ProcessPayroll.php
├── GeneratePayslips.php
├── SendPayslipEmails.php
├── AccrueLeaveBalances.php
```

### 7. NOTIFICATIONS (4 files)
```
app/Notifications/
├── PayslipReadyNotification.php
├── LeaveRequestNotification.php
├── LeaveApprovedNotification.php
├── AttendanceReminderNotification.php
```

### 8. VIEWS (2 files)
```
resources/views/payroll/
├── payslip-pdf.blade.php
├── payslip-email.blade.php
```

### 9. TESTS (5 files)
```
tests/Feature/
├── EmployeeTest.php
├── PayrollTest.php
├── LeaveTest.php
├── AttendanceTest.php
├── PayrollCalculationTest.php
```

---

## 📝 KEY CONTROLLERS

### EmployeeController.php
```php
public function index(Request $request);      // List employees
public function store(StoreEmployeeRequest $request);
public function show(Employee $employee);
public function update(UpdateEmployeeRequest $request, Employee $employee);
public function destroy(Employee $employee);
public function documents(Employee $employee);
public function uploadDocument(Request $request, Employee $employee);
public function salary(Employee $employee);   // Salary structure
public function updateSalary(Request $request, Employee $employee);
public function leaveBalances(Employee $employee);
public function attendance(Employee $employee, Request $request);
public function loans(Employee $employee);
public function orgChart();                   // Organization chart
public function directory();                  // Employee directory
```

### PayrollController.php
```php
public function periods(Request $request);    // List pay periods
public function createPeriod(Request $request);
public function runs(Request $request);       // List payroll runs
public function createRun(CreatePayrollRunRequest $request);
public function showRun(PayrollRun $run);
public function calculateRun(PayrollRun $run);
public function approveRun(PayrollRun $run);
public function processRun(PayrollRun $run);  // Mark as paid
public function payslip(PayrollItem $item);   // Individual payslip
public function payslipPdf(PayrollItem $item);
public function sendPayslips(PayrollRun $run);
public function summary(Request $request);    // Payroll summary
```

### LeaveController.php
```php
public function types();                      // Leave types
public function storeType(StoreLeaveTypeRequest $request);
public function requests(Request $request);   // Leave requests
public function myRequests();                 // My leave requests
public function myBalances();                 // My balances
public function store(StoreLeaveRequestRequest $request);
public function approve(LeaveRequest $request);
public function reject(Request $httpRequest, LeaveRequest $request);
public function cancel(LeaveRequest $request);
public function calendar(Request $request);   // Team leave calendar
```

### AttendanceController.php
```php
public function index(Request $request);      // List attendance
public function store(StoreAttendanceRequest $request);
public function checkIn(Request $request);    // Clock in
public function checkOut();                   // Clock out
public function myAttendance(Request $request);
public function bulkStore(BulkAttendanceRequest $request);
public function report(Request $request);     // Attendance report
public function summary(Request $request);    // Monthly summary
```

---

## 🔗 ROUTES

```php
// routes/api/hr.php

Route::middleware('auth:sanctum')->group(function () {
    // Employees
    Route::apiResource('employees', EmployeeController::class);
    Route::get('employees/{employee}/documents', [EmployeeController::class, 'documents']);
    Route::post('employees/{employee}/documents', [EmployeeController::class, 'uploadDocument']);
    Route::get('employees/{employee}/salary', [EmployeeController::class, 'salary']);
    Route::put('employees/{employee}/salary', [EmployeeController::class, 'updateSalary']);
    Route::get('employees/{employee}/leave-balances', [EmployeeController::class, 'leaveBalances']);
    Route::get('employees/{employee}/attendance', [EmployeeController::class, 'attendance']);
    Route::get('employees/{employee}/loans', [EmployeeController::class, 'loans']);
    Route::get('org-chart', [EmployeeController::class, 'orgChart']);
    Route::get('employee-directory', [EmployeeController::class, 'directory']);
    
    // Payroll
    Route::get('pay-periods', [PayrollController::class, 'periods']);
    Route::post('pay-periods', [PayrollController::class, 'createPeriod']);
    Route::get('payroll-runs', [PayrollController::class, 'runs']);
    Route::post('payroll-runs', [PayrollController::class, 'createRun']);
    Route::get('payroll-runs/{run}', [PayrollController::class, 'showRun']);
    Route::post('payroll-runs/{run}/calculate', [PayrollController::class, 'calculateRun']);
    Route::post('payroll-runs/{run}/approve', [PayrollController::class, 'approveRun']);
    Route::post('payroll-runs/{run}/process', [PayrollController::class, 'processRun']);
    Route::post('payroll-runs/{run}/send-payslips', [PayrollController::class, 'sendPayslips']);
    Route::get('payroll-items/{item}/payslip', [PayrollController::class, 'payslipPdf']);
    Route::get('payroll-summary', [PayrollController::class, 'summary']);
    
    // Salary Components
    Route::apiResource('salary-components', SalaryComponentController::class);
    
    // Leave
    Route::get('leave-types', [LeaveController::class, 'types']);
    Route::post('leave-types', [LeaveController::class, 'storeType']);
    Route::get('leave-requests', [LeaveController::class, 'requests']);
    Route::post('leave-requests', [LeaveController::class, 'store']);
    Route::post('leave-requests/{request}/approve', [LeaveController::class, 'approve']);
    Route::post('leave-requests/{request}/reject', [LeaveController::class, 'reject']);
    Route::post('leave-requests/{request}/cancel', [LeaveController::class, 'cancel']);
    Route::get('my/leave-requests', [LeaveController::class, 'myRequests']);
    Route::get('my/leave-balances', [LeaveController::class, 'myBalances']);
    Route::get('leave-calendar', [LeaveController::class, 'calendar']);
    
    // Attendance
    Route::get('attendance', [AttendanceController::class, 'index']);
    Route::post('attendance', [AttendanceController::class, 'store']);
    Route::post('attendance/check-in', [AttendanceController::class, 'checkIn']);
    Route::post('attendance/check-out', [AttendanceController::class, 'checkOut']);
    Route::post('attendance/bulk', [AttendanceController::class, 'bulkStore']);
    Route::get('my/attendance', [AttendanceController::class, 'myAttendance']);
    Route::get('attendance-report', [AttendanceController::class, 'report']);
    
    // Loans
    Route::apiResource('employee-loans', EmployeeLoanController::class);
    Route::post('employee-loans/{loan}/approve', [EmployeeLoanController::class, 'approve']);
    Route::post('employee-loans/{loan}/payment', [EmployeeLoanController::class, 'recordPayment']);
    
    // Holidays
    Route::apiResource('holidays', HolidayController::class);
    
    // HR Reports
    Route::get('hr/reports/headcount', [HRReportController::class, 'headcount']);
    Route::get('hr/reports/turnover', [HRReportController::class, 'turnover']);
    Route::get('hr/reports/leave-summary', [HRReportController::class, 'leaveSummary']);
    Route::get('hr/reports/payroll-summary', [HRReportController::class, 'payrollSummary']);
});
```

---

## 🇧🇭 BAHRAIN-SPECIFIC

1. **Employee Fields**: CPR number, visa/work permit tracking
2. **Leave Types**: Annual (30 days), Sick, Maternity, Paternity, Hajj
3. **Social Insurance**: GOSI contributions (employer + employee)
4. **Working Week**: Sunday-Thursday (Friday-Saturday off)
5. **Currency**: BHD for all salary calculations

---

## ✅ COMPLETION CHECKLIST

- [ ] Employee CRUD + documents
- [ ] Salary structure management
- [ ] Payroll run + calculation
- [ ] Payslip PDF generation
- [ ] Leave types + balances
- [ ] Leave requests + approval
- [ ] Attendance tracking
- [ ] Employee loans
- [ ] Holiday calendar
- [ ] HR reports
- [ ] All routes defined
- [ ] All tests passing

---

## 📤 OUTPUT

Save to: `/home/claude/business-platform/`
Update: `/home/claude/business-platform/docs/PROGRESS_BOT_6.md`
