# 🤖 BOT 8: MARKETING / AUTOMATION MODULE

## 📋 YOUR RESPONSIBILITY
Build workflows, campaigns, forms, landing pages, and marketing automation.

---

## 📁 FILES TO CREATE

### 1. MODELS (12 files)
```
app/Models/
├── Workflow.php ✅ EXISTS - ENHANCE
├── WorkflowEnrollment.php
├── WorkflowLog.php
├── WorkflowStep.php
├── Campaign.php
├── CampaignRecipient.php
├── Form.php
├── FormSubmission.php
├── FormField.php
├── LandingPage.php
├── Funnel.php
├── FunnelStep.php
```

### 2. CONTROLLERS (7 files)
```
app/Http/Controllers/Api/
├── WorkflowController.php
├── CampaignController.php
├── FormController.php
├── FormSubmissionController.php
├── LandingPageController.php
├── FunnelController.php
├── MarketingDashboardController.php
```

### 3. SERVICES (6 files)
```
app/Services/
├── WorkflowService.php
├── WorkflowEngineService.php
├── CampaignService.php
├── FormService.php
├── LandingPageService.php
├── FunnelService.php
```

### 4. FORM REQUESTS (10 files)
```
app/Http/Requests/
├── Workflow/StoreWorkflowRequest.php
├── Workflow/UpdateWorkflowRequest.php
├── Campaign/StoreCampaignRequest.php
├── Campaign/UpdateCampaignRequest.php
├── Campaign/ScheduleCampaignRequest.php
├── Form/StoreFormRequest.php
├── Form/UpdateFormRequest.php
├── LandingPage/StoreLandingPageRequest.php
├── Funnel/StoreFunnelRequest.php
├── Funnel/UpdateFunnelRequest.php
```

### 5. ROUTES (1 file)
```
routes/api/marketing.php
```

### 6. JOBS (6 files)
```
app/Jobs/
├── ProcessWorkflowStep.php
├── SendCampaign.php
├── ProcessCampaignRecipient.php
├── ProcessFormSubmission.php
├── CleanupExpiredForms.php
├── CalculateCampaignStats.php
```

### 7. EVENTS & LISTENERS
```
app/Events/
├── WorkflowTriggered.php
├── FormSubmitted.php
├── CampaignSent.php

app/Listeners/
├── EnrollInWorkflow.php
├── ProcessWorkflowTrigger.php
├── CreateContactFromForm.php
├── NotifyOnFormSubmission.php
```

### 8. TESTS (5 files)
```
tests/Feature/
├── WorkflowTest.php
├── CampaignTest.php
├── FormTest.php
├── LandingPageTest.php
├── FunnelTest.php
```

---

## 📝 KEY CONTROLLERS

### WorkflowController.php
```php
public function index(Request $request);      // List workflows
public function store(StoreWorkflowRequest $request);
public function show(Workflow $workflow);
public function update(UpdateWorkflowRequest $request, Workflow $workflow);
public function destroy(Workflow $workflow);
public function duplicate(Workflow $workflow);
public function activate(Workflow $workflow);
public function deactivate(Workflow $workflow);
public function enrollments(Workflow $workflow);  // List enrollments
public function enroll(Workflow $workflow, Request $request);  // Manual enroll
public function unenroll(Workflow $workflow, Contact $contact);
public function logs(Workflow $workflow);     // Execution logs
public function stats(Workflow $workflow);    // Performance stats
public function test(Workflow $workflow);     // Test workflow
```

### CampaignController.php
```php
public function index(Request $request);      // List campaigns
public function store(StoreCampaignRequest $request);
public function show(Campaign $campaign);
public function update(UpdateCampaignRequest $request, Campaign $campaign);
public function destroy(Campaign $campaign);
public function duplicate(Campaign $campaign);
public function schedule(ScheduleCampaignRequest $request, Campaign $campaign);
public function send(Campaign $campaign);     // Send now
public function pause(Campaign $campaign);
public function resume(Campaign $campaign);
public function cancel(Campaign $campaign);
public function recipients(Campaign $campaign);
public function stats(Campaign $campaign);    // Open/click rates
public function preview(Campaign $campaign);
public function testSend(Campaign $campaign, Request $request);
```

### FormController.php
```php
public function index(Request $request);      // List forms
public function store(StoreFormRequest $request);
public function show(Form $form);
public function update(UpdateFormRequest $request, Form $form);
public function destroy(Form $form);
public function duplicate(Form $form);
public function publish(Form $form);
public function unpublish(Form $form);
public function submissions(Form $form);
public function exportSubmissions(Form $form);
public function embed(Form $form);            // Get embed code
public function stats(Form $form);
```

### Public Form Submission (no auth)
```php
// PublicFormController.php
public function show(string $formId);         // Get form fields
public function submit(string $formId, Request $request);  // Submit form
```

---

## 🔗 ROUTES

```php
// routes/api/marketing.php

Route::middleware('auth:sanctum')->group(function () {
    // Workflows
    Route::apiResource('workflows', WorkflowController::class);
    Route::post('workflows/{workflow}/duplicate', [WorkflowController::class, 'duplicate']);
    Route::post('workflows/{workflow}/activate', [WorkflowController::class, 'activate']);
    Route::post('workflows/{workflow}/deactivate', [WorkflowController::class, 'deactivate']);
    Route::get('workflows/{workflow}/enrollments', [WorkflowController::class, 'enrollments']);
    Route::post('workflows/{workflow}/enroll', [WorkflowController::class, 'enroll']);
    Route::delete('workflows/{workflow}/enrollments/{contact}', [WorkflowController::class, 'unenroll']);
    Route::get('workflows/{workflow}/logs', [WorkflowController::class, 'logs']);
    Route::get('workflows/{workflow}/stats', [WorkflowController::class, 'stats']);
    Route::post('workflows/{workflow}/test', [WorkflowController::class, 'test']);
    
    // Campaigns
    Route::apiResource('campaigns', CampaignController::class);
    Route::post('campaigns/{campaign}/duplicate', [CampaignController::class, 'duplicate']);
    Route::post('campaigns/{campaign}/schedule', [CampaignController::class, 'schedule']);
    Route::post('campaigns/{campaign}/send', [CampaignController::class, 'send']);
    Route::post('campaigns/{campaign}/pause', [CampaignController::class, 'pause']);
    Route::post('campaigns/{campaign}/resume', [CampaignController::class, 'resume']);
    Route::post('campaigns/{campaign}/cancel', [CampaignController::class, 'cancel']);
    Route::get('campaigns/{campaign}/recipients', [CampaignController::class, 'recipients']);
    Route::get('campaigns/{campaign}/stats', [CampaignController::class, 'stats']);
    Route::get('campaigns/{campaign}/preview', [CampaignController::class, 'preview']);
    Route::post('campaigns/{campaign}/test', [CampaignController::class, 'testSend']);
    
    // Forms
    Route::apiResource('forms', FormController::class);
    Route::post('forms/{form}/duplicate', [FormController::class, 'duplicate']);
    Route::post('forms/{form}/publish', [FormController::class, 'publish']);
    Route::post('forms/{form}/unpublish', [FormController::class, 'unpublish']);
    Route::get('forms/{form}/submissions', [FormController::class, 'submissions']);
    Route::get('forms/{form}/submissions/export', [FormController::class, 'exportSubmissions']);
    Route::get('forms/{form}/embed', [FormController::class, 'embed']);
    Route::get('forms/{form}/stats', [FormController::class, 'stats']);
    
    // Form Submissions
    Route::get('form-submissions', [FormSubmissionController::class, 'index']);
    Route::get('form-submissions/{submission}', [FormSubmissionController::class, 'show']);
    Route::delete('form-submissions/{submission}', [FormSubmissionController::class, 'destroy']);
    
    // Landing Pages
    Route::apiResource('landing-pages', LandingPageController::class);
    Route::post('landing-pages/{page}/publish', [LandingPageController::class, 'publish']);
    Route::post('landing-pages/{page}/unpublish', [LandingPageController::class, 'unpublish']);
    Route::get('landing-pages/{page}/stats', [LandingPageController::class, 'stats']);
    
    // Funnels
    Route::apiResource('funnels', FunnelController::class);
    Route::get('funnels/{funnel}/stats', [FunnelController::class, 'stats']);
    
    // Marketing Dashboard
    Route::get('marketing/dashboard', [MarketingDashboardController::class, 'index']);
    Route::get('marketing/reports/campaigns', [MarketingDashboardController::class, 'campaignReport']);
    Route::get('marketing/reports/forms', [MarketingDashboardController::class, 'formReport']);
});

// Public (no auth)
Route::get('f/{formId}', [PublicFormController::class, 'show']);
Route::post('f/{formId}', [PublicFormController::class, 'submit']);
Route::get('p/{pageSlug}', [PublicLandingPageController::class, 'show']);
Route::get('t/{pixelId}', [TrackingController::class, 'pixel']);  // Email tracking pixel
Route::get('c/{linkId}', [TrackingController::class, 'click']);   // Click tracking
```

---

## 🔄 WORKFLOW TRIGGERS

```
- contact_created
- contact_updated
- tag_added
- tag_removed
- form_submitted
- opportunity_created
- opportunity_stage_changed
- opportunity_won
- opportunity_lost
- invoice_paid
- task_completed
- appointment_booked
- manual
- scheduled
- webhook
```

## 🔄 WORKFLOW ACTIONS

```
- send_email
- send_sms
- send_whatsapp
- add_tag
- remove_tag
- update_contact
- create_task
- create_opportunity
- move_opportunity_stage
- wait (delay)
- if_else (condition)
- go_to (jump to step)
- webhook (call external URL)
- notify_user
- assign_owner
```

---

## ✅ COMPLETION CHECKLIST

- [ ] Workflow builder + engine
- [ ] Workflow triggers + actions
- [ ] Campaign CRUD + scheduling
- [ ] Email/SMS campaign sending
- [ ] Form builder
- [ ] Form submissions + webhooks
- [ ] Landing page builder
- [ ] Funnel management
- [ ] Tracking (opens, clicks)
- [ ] Analytics/stats
- [ ] All routes defined
- [ ] All tests passing

---

## 📤 OUTPUT

Save to: `/home/claude/business-platform/`
Update: `/home/claude/business-platform/docs/PROGRESS_BOT_8.md`
