# 🤖 BOT 9: FRONTEND - CORE / CRM

## 📋 YOUR RESPONSIBILITY
Build Vue.js/React frontend for authentication, dashboard, CRM (contacts, pipelines, opportunities).

---

## 🛠️ TECH STACK

- **Framework**: Vue 3 + Composition API OR React 18
- **State**: Pinia (Vue) or Zustand (React)
- **UI**: Tailwind CSS + Headless UI
- **HTTP**: Axios
- **Routing**: Vue Router / React Router
- **Forms**: VeeValidate / React Hook Form
- **Tables**: TanStack Table
- **Charts**: Chart.js / Recharts
- **Drag & Drop**: Vue Draggable / React DnD

---

## 📁 FILES TO CREATE

### 1. LAYOUTS (4 files)
```
resources/js/Layouts/
├── AuthLayout.vue           # Login/register layout
├── AppLayout.vue            # Main app with sidebar
├── SettingsLayout.vue       # Settings pages
├── PublicLayout.vue         # Public pages (forms, invoices)
```

### 2. PAGES - AUTH (6 files)
```
resources/js/Pages/Auth/
├── Login.vue
├── Register.vue
├── ForgotPassword.vue
├── ResetPassword.vue
├── VerifyEmail.vue
├── TwoFactor.vue
```

### 3. PAGES - DASHBOARD (3 files)
```
resources/js/Pages/Dashboard/
├── Index.vue                # Main dashboard
├── Widgets/
│   ├── StatsWidget.vue
│   ├── RevenueChart.vue
│   ├── TasksWidget.vue
│   ├── ActivitiesWidget.vue
│   └── PipelineWidget.vue
```

### 4. PAGES - CONTACTS (8 files)
```
resources/js/Pages/Contacts/
├── Index.vue                # Contact list
├── Show.vue                 # Contact detail
├── Create.vue               # Create contact
├── Edit.vue                 # Edit contact
├── Import.vue               # Import contacts
├── Partials/
│   ├── ContactTable.vue
│   ├── ContactFilters.vue
│   ├── ContactTimeline.vue
│   ├── ContactForm.vue
│   └── MergeContactsModal.vue
```

### 5. PAGES - PIPELINES (6 files)
```
resources/js/Pages/Pipelines/
├── Index.vue                # Pipeline list
├── Kanban.vue               # Kanban board
├── Show.vue                 # Pipeline detail
├── Partials/
│   ├── KanbanColumn.vue
│   ├── OpportunityCard.vue
│   ├── OpportunityModal.vue
│   ├── StageEditor.vue
│   └── PipelineSettings.vue
```

### 6. PAGES - OPPORTUNITIES (5 files)
```
resources/js/Pages/Opportunities/
├── Index.vue                # List view
├── Show.vue                 # Detail view
├── Create.vue
├── Edit.vue
├── Partials/
│   ├── OpportunityForm.vue
│   ├── OpportunityProducts.vue
│   └── OpportunityActivities.vue
```

### 7. PAGES - ACTIVITIES (4 files)
```
resources/js/Pages/Activities/
├── Index.vue
├── Create.vue
├── Partials/
│   ├── ActivityForm.vue
│   ├── ActivityList.vue
│   └── ActivityCalendar.vue
```

### 8. COMPONENTS - UI (20 files)
```
resources/js/Components/
├── App/
│   ├── Sidebar.vue
│   ├── Navbar.vue
│   ├── UserDropdown.vue
│   ├── NotificationDropdown.vue
│   └── SearchPalette.vue
├── Form/
│   ├── Input.vue
│   ├── Select.vue
│   ├── Textarea.vue
│   ├── Checkbox.vue
│   ├── Radio.vue
│   ├── DatePicker.vue
│   ├── FilePicker.vue
│   ├── TagInput.vue
│   └── CurrencyInput.vue
├── UI/
│   ├── Button.vue
│   ├── Modal.vue
│   ├── Dropdown.vue
│   ├── Badge.vue
│   ├── Avatar.vue
│   ├── Card.vue
│   ├── Table.vue
│   ├── Pagination.vue
│   ├── EmptyState.vue
│   ├── Loading.vue
│   ├── Alert.vue
│   └── Tabs.vue
```

### 9. COMPOSABLES/HOOKS (8 files)
```
resources/js/Composables/
├── useAuth.js
├── useApi.js
├── useContacts.js
├── usePipelines.js
├── useOpportunities.js
├── useActivities.js
├── useNotifications.js
└── useFilters.js
```

### 10. STORES (6 files)
```
resources/js/Stores/
├── auth.js
├── contacts.js
├── pipelines.js
├── opportunities.js
├── activities.js
└── notifications.js
```

### 11. API SERVICES (6 files)
```
resources/js/Services/
├── api.js                   # Axios instance
├── authService.js
├── contactService.js
├── pipelineService.js
├── opportunityService.js
└── activityService.js
```

### 12. ROUTER (1 file)
```
resources/js/router/
├── index.js                 # Route definitions
```

---

## 📝 PAGE SPECIFICATIONS

### Login.vue
```vue
<template>
  <AuthLayout>
    <div class="max-w-md mx-auto">
      <h1>Sign in to your account</h1>
      <form @submit.prevent="login">
        <Input v-model="form.email" label="Email" type="email" />
        <Input v-model="form.password" label="Password" type="password" />
        <Checkbox v-model="form.remember" label="Remember me" />
        <Button type="submit" :loading="loading">Sign in</Button>
      </form>
      <p>Don't have an account? <router-link to="/register">Sign up</router-link></p>
      <router-link to="/forgot-password">Forgot password?</router-link>
    </div>
  </AuthLayout>
</template>
```

### Dashboard/Index.vue
```vue
<template>
  <AppLayout title="Dashboard">
    <div class="grid grid-cols-4 gap-6">
      <StatsWidget title="Total Contacts" :value="stats.contacts" icon="users" />
      <StatsWidget title="Open Opportunities" :value="stats.opportunities" icon="target" />
      <StatsWidget title="Revenue (MTD)" :value="stats.revenue" icon="dollar" />
      <StatsWidget title="Tasks Due" :value="stats.tasksDue" icon="check" />
    </div>
    
    <div class="grid grid-cols-2 gap-6 mt-6">
      <RevenueChart :data="revenueData" />
      <PipelineWidget :data="pipelineData" />
    </div>
    
    <div class="grid grid-cols-2 gap-6 mt-6">
      <TasksWidget :tasks="upcomingTasks" />
      <ActivitiesWidget :activities="recentActivities" />
    </div>
  </AppLayout>
</template>
```

### Contacts/Index.vue
```vue
<template>
  <AppLayout title="Contacts">
    <div class="flex justify-between mb-6">
      <ContactFilters v-model="filters" />
      <div class="flex gap-2">
        <Button @click="importModal = true">Import</Button>
        <Button @click="$router.push('/contacts/create')">Add Contact</Button>
      </div>
    </div>
    
    <ContactTable 
      :contacts="contacts" 
      :loading="loading"
      @select="selectedContacts = $event"
      @sort="handleSort"
    />
    
    <Pagination :meta="meta" @change="fetchContacts" />
  </AppLayout>
</template>
```

### Pipelines/Kanban.vue
```vue
<template>
  <AppLayout title="Sales Pipeline">
    <div class="flex justify-between mb-6">
      <Select v-model="selectedPipeline" :options="pipelines" />
      <Button @click="opportunityModal = true">Add Opportunity</Button>
    </div>
    
    <div class="flex gap-4 overflow-x-auto pb-4">
      <KanbanColumn
        v-for="stage in stages"
        :key="stage.id"
        :stage="stage"
        :opportunities="stage.opportunities"
        @drop="handleDrop"
        @click="openOpportunity"
      />
    </div>
    
    <OpportunityModal 
      v-model="opportunityModal" 
      :opportunity="selectedOpportunity"
      @save="saveOpportunity"
    />
  </AppLayout>
</template>
```

---

## 🎨 DESIGN SYSTEM

### Colors (Setup in Bahrain brand)
```js
// tailwind.config.js
colors: {
  primary: '#1B5E20',    // Green
  secondary: '#2196F3',  // Blue
  accent: '#E91E63',     // Red
  warning: '#FFC107',    // Golden
}
```

### Typography
```css
/* Base styles */
body { @apply text-gray-900 bg-gray-50; }
h1 { @apply text-2xl font-bold; }
h2 { @apply text-xl font-semibold; }
h3 { @apply text-lg font-medium; }
```

---

## 🔗 ROUTES

```js
// resources/js/router/index.js

const routes = [
  // Auth
  { path: '/login', component: Login, meta: { guest: true } },
  { path: '/register', component: Register, meta: { guest: true } },
  { path: '/forgot-password', component: ForgotPassword, meta: { guest: true } },
  { path: '/reset-password/:token', component: ResetPassword, meta: { guest: true } },
  
  // Dashboard
  { path: '/', component: Dashboard, meta: { auth: true } },
  
  // Contacts
  { path: '/contacts', component: ContactsIndex, meta: { auth: true } },
  { path: '/contacts/create', component: ContactsCreate, meta: { auth: true } },
  { path: '/contacts/:id', component: ContactsShow, meta: { auth: true } },
  { path: '/contacts/:id/edit', component: ContactsEdit, meta: { auth: true } },
  { path: '/contacts/import', component: ContactsImport, meta: { auth: true } },
  
  // Pipelines & Opportunities
  { path: '/pipelines', component: PipelinesIndex, meta: { auth: true } },
  { path: '/pipelines/:id/kanban', component: PipelinesKanban, meta: { auth: true } },
  { path: '/opportunities', component: OpportunitiesIndex, meta: { auth: true } },
  { path: '/opportunities/create', component: OpportunitiesCreate, meta: { auth: true } },
  { path: '/opportunities/:id', component: OpportunitiesShow, meta: { auth: true } },
  
  // Activities
  { path: '/activities', component: ActivitiesIndex, meta: { auth: true } },
];
```

---

## ✅ COMPLETION CHECKLIST

- [ ] Auth pages (login, register, forgot)
- [ ] Main app layout + sidebar
- [ ] Dashboard with widgets
- [ ] Contact list + CRUD
- [ ] Contact detail + timeline
- [ ] Pipeline list
- [ ] Kanban board (drag & drop)
- [ ] Opportunity CRUD
- [ ] Activity logging
- [ ] UI components library
- [ ] API services
- [ ] State management
- [ ] Responsive design

---

## 📤 OUTPUT

Save to: `/home/claude/business-platform/resources/js/`
Update: `/home/claude/business-platform/docs/PROGRESS_BOT_9.md`
