<?php

use App\Http\Controllers\Api\{
    ActivityController, AttendanceController, AuthController, BankAccountController,
    BillController, CampaignController, ChartOfAccountController, ContactController,
    DashboardController, DepartmentController, EmailController, EmployeeController,
    EstimateController, ExpenseController, FormController, ImportExportController,
    InvoiceController, JournalEntryController, LeaveController, LeaveRequestController,
    MilestoneController, NotificationController, OpportunityController, PaymentController,
    PayrollController, PipelineController, ProductController, ProjectController,
    RecurringInvoiceController, ReportController, SegmentController, SettingsController,
    SmsController, TagController, TaskController, TeamController, TenantController,
    TimeEntryController, UserController, VendorController, WebhookController,
    WhatsappController, WorkflowController
};
use Illuminate\Support\Facades\Route;

// Public routes
Route::post('/auth/register', [AuthController::class, 'register']);
Route::post('/auth/login', [AuthController::class, 'login']);
Route::post('/auth/forgot-password', [AuthController::class, 'forgotPassword']);
Route::post('/auth/reset-password', [AuthController::class, 'resetPassword']);

// Form submissions (public)
Route::get('/forms/{slug}', [FormController::class, 'showBySlug']);
Route::post('/forms/{slug}/submit', [FormController::class, 'submit']);

// Webhooks (public)
Route::post('/webhooks/{provider}', [WebhookController::class, 'handle']);

// Protected routes
Route::middleware(['auth:sanctum', 'tenant'])->group(function () {
    
    // Auth
    Route::post('/auth/logout', [AuthController::class, 'logout']);
    Route::get('/auth/me', [AuthController::class, 'me']);
    Route::put('/auth/profile', [AuthController::class, 'updateProfile']);
    Route::put('/auth/password', [AuthController::class, 'changePassword']);

    // Dashboard
    Route::get('/dashboard/stats', [DashboardController::class, 'stats']);
    Route::get('/dashboard/activities', [DashboardController::class, 'activities']);
    Route::get('/dashboard/pipeline', [DashboardController::class, 'pipeline']);
    Route::get('/dashboard/revenue-chart', [DashboardController::class, 'revenueChart']);

    // CRM - Contacts
    Route::apiResource('contacts', ContactController::class);
    Route::post('/contacts/import', [ContactController::class, 'import']);
    Route::get('/contacts/export', [ContactController::class, 'export']);
    Route::post('/contacts/{contact}/tags', [ContactController::class, 'addTags']);
    Route::delete('/contacts/{contact}/tags/{tag}', [ContactController::class, 'removeTag']);
    Route::get('/contacts/{contact}/activities', [ContactController::class, 'activities']);
    Route::get('/contacts/{contact}/notes', [ContactController::class, 'notes']);
    Route::post('/contacts/{contact}/notes', [ContactController::class, 'addNote']);

    // CRM - Segments
    Route::apiResource('segments', SegmentController::class);
    Route::post('/segments/{segment}/contacts', [SegmentController::class, 'addContacts']);
    Route::delete('/segments/{segment}/contacts', [SegmentController::class, 'removeContacts']);
    Route::post('/segments/{segment}/refresh', [SegmentController::class, 'refresh']);
    Route::post('/segments/{segment}/duplicate', [SegmentController::class, 'duplicate']);

    // CRM - Tags
    Route::apiResource('tags', TagController::class);

    // CRM - Pipelines & Opportunities
    Route::apiResource('pipelines', PipelineController::class);
    Route::apiResource('opportunities', OpportunityController::class);
    Route::post('/opportunities/{opportunity}/move', [OpportunityController::class, 'moveStage']);
    Route::post('/opportunities/{opportunity}/won', [OpportunityController::class, 'markWon']);
    Route::post('/opportunities/{opportunity}/lost', [OpportunityController::class, 'markLost']);

    // CRM - Forms
    Route::apiResource('forms', FormController::class);
    Route::get('/forms/{form}/submissions', [FormController::class, 'submissions']);
    Route::post('/forms/{form}/duplicate', [FormController::class, 'duplicate']);

    // Communication - Email
    Route::get('/emails', [EmailController::class, 'index']);
    Route::post('/emails', [EmailController::class, 'send']);
    Route::get('/emails/{email}', [EmailController::class, 'show']);
    Route::get('/email-templates', [EmailController::class, 'templates']);
    Route::apiResource('email-templates', EmailController::class)->only(['store', 'update', 'destroy']);

    // Communication - SMS
    Route::get('/sms', [SmsController::class, 'index']);
    Route::post('/sms', [SmsController::class, 'send']);

    // Communication - WhatsApp
    Route::get('/whatsapp', [WhatsappController::class, 'index']);
    Route::post('/whatsapp', [WhatsappController::class, 'send']);

    // Communication - Campaigns
    Route::apiResource('campaigns', CampaignController::class);
    Route::post('/campaigns/{campaign}/send', [CampaignController::class, 'send']);
    Route::post('/campaigns/{campaign}/schedule', [CampaignController::class, 'schedule']);
    Route::post('/campaigns/{campaign}/duplicate', [CampaignController::class, 'duplicate']);
    Route::get('/campaigns/{campaign}/stats', [CampaignController::class, 'stats']);

    // Automation - Workflows
    Route::apiResource('workflows', WorkflowController::class);
    Route::post('/workflows/{workflow}/activate', [WorkflowController::class, 'activate']);
    Route::post('/workflows/{workflow}/deactivate', [WorkflowController::class, 'deactivate']);
    Route::post('/workflows/{workflow}/enroll', [WorkflowController::class, 'enroll']);
    Route::get('/workflows/stats', [WorkflowController::class, 'stats']);

    // Accounting - Invoices
    Route::apiResource('invoices', InvoiceController::class);
    Route::post('/invoices/{invoice}/send', [InvoiceController::class, 'send']);
    Route::post('/invoices/{invoice}/payment', [InvoiceController::class, 'recordPayment']);
    Route::get('/invoices/{invoice}/pdf', [InvoiceController::class, 'pdf']);
    Route::post('/invoices/{invoice}/duplicate', [InvoiceController::class, 'duplicate']);

    // Accounting - Recurring Invoices
    Route::apiResource('recurring-invoices', RecurringInvoiceController::class);
    Route::post('/recurring-invoices/{recurringInvoice}/generate', [RecurringInvoiceController::class, 'generate']);
    Route::get('/recurring-invoices/stats', [RecurringInvoiceController::class, 'stats']);

    // Accounting - Estimates/Quotes
    Route::apiResource('estimates', EstimateController::class);
    Route::post('/estimates/{estimate}/send', [EstimateController::class, 'send']);
    Route::post('/estimates/{estimate}/convert', [EstimateController::class, 'convertToInvoice']);
    Route::post('/estimates/{estimate}/accept', [EstimateController::class, 'accept']);
    Route::post('/estimates/{estimate}/reject', [EstimateController::class, 'reject']);

    // Accounting - Payments
    Route::apiResource('payments', PaymentController::class);

    // Accounting - Expenses
    Route::apiResource('expenses', ExpenseController::class);
    Route::post('/expenses/{expense}/approve', [ExpenseController::class, 'approve']);
    Route::post('/expenses/{expense}/reject', [ExpenseController::class, 'reject']);

    // Accounting - Bills
    Route::apiResource('bills', BillController::class);
    Route::post('/bills/{bill}/payment', [BillController::class, 'recordPayment']);

    // Accounting - Vendors
    Route::apiResource('vendors', VendorController::class);

    // Accounting - Products/Services
    Route::apiResource('products', ProductController::class);

    // Accounting - Chart of Accounts
    Route::apiResource('chart-of-accounts', ChartOfAccountController::class);

    // Accounting - Journal Entries
    Route::apiResource('journal-entries', JournalEntryController::class);

    // Accounting - Bank Accounts
    Route::apiResource('bank-accounts', BankAccountController::class);
    Route::get('/bank-accounts/{bankAccount}/transactions', [BankAccountController::class, 'transactions']);
    Route::post('/bank-accounts/{bankAccount}/sync', [BankAccountController::class, 'sync']);
    Route::post('/bank-accounts/{bankAccount}/reconcile', [BankAccountController::class, 'reconcile']);

    // Task Management - Projects
    Route::apiResource('projects', ProjectController::class);
    Route::get('/projects/{project}/tasks', [ProjectController::class, 'tasks']);
    Route::post('/projects/{project}/tasks', [TaskController::class, 'store']);
    Route::get('/projects/{project}/members', [ProjectController::class, 'members']);
    Route::post('/projects/{project}/members', [ProjectController::class, 'addMember']);

    // Task Management - Tasks
    Route::apiResource('tasks', TaskController::class);
    Route::post('/tasks/{task}/complete', [TaskController::class, 'complete']);
    Route::put('/tasks/{task}/status', [TaskController::class, 'updateStatus']);
    Route::post('/tasks/{task}/comments', [TaskController::class, 'addComment']);
    Route::get('/tasks/{task}/comments', [TaskController::class, 'comments']);
    Route::post('/tasks/{task}/attachments', [TaskController::class, 'addAttachment']);
    Route::post('/tasks/{task}/assignees', [TaskController::class, 'addAssignee']);

    // Task Management - Milestones
    Route::apiResource('milestones', MilestoneController::class);
    Route::post('/milestones/{milestone}/complete', [MilestoneController::class, 'complete']);

    // Time Tracking
    Route::apiResource('time-entries', TimeEntryController::class);
    Route::post('/time-entries/start', [TimeEntryController::class, 'start']);
    Route::post('/time-entries/{timeEntry}/stop', [TimeEntryController::class, 'stop']);
    Route::get('/time-entries/running', [TimeEntryController::class, 'running']);
    Route::get('/time-entries/summary', [TimeEntryController::class, 'summary']);

    // HR - Employees
    Route::apiResource('employees', EmployeeController::class);
    Route::post('/employees/{employee}/salary', [EmployeeController::class, 'updateSalary']);
    Route::post('/employees/{employee}/terminate', [EmployeeController::class, 'terminate']);
    Route::get('/employees/{employee}/payslips', [EmployeeController::class, 'payslips']);

    // HR - Departments
    Route::apiResource('departments', DepartmentController::class);
    Route::get('/departments/hierarchy', [DepartmentController::class, 'hierarchy']);
    Route::get('/departments/stats', [DepartmentController::class, 'stats']);

    // HR - Attendance
    Route::get('/attendance', [AttendanceController::class, 'index']);
    Route::post('/attendance', [AttendanceController::class, 'store']);
    Route::post('/attendance/check-in', [AttendanceController::class, 'checkIn']);
    Route::post('/attendance/check-out', [AttendanceController::class, 'checkOut']);
    Route::get('/attendance/today', [AttendanceController::class, 'today']);

    // HR - Leave Management
    Route::get('/leave-types', [LeaveController::class, 'types']);
    Route::apiResource('leave-requests', LeaveRequestController::class);
    Route::post('/leave-requests/{leaveRequest}/approve', [LeaveRequestController::class, 'approve']);
    Route::post('/leave-requests/{leaveRequest}/reject', [LeaveRequestController::class, 'reject']);
    Route::post('/leave-requests/{leaveRequest}/cancel', [LeaveRequestController::class, 'cancel']);
    Route::get('/leave-balances', [LeaveRequestController::class, 'balances']);

    // HR - Payroll
    Route::apiResource('payroll-runs', PayrollController::class);
    Route::post('/payroll-runs/{payrollRun}/process', [PayrollController::class, 'process']);
    Route::post('/payroll-runs/{payrollRun}/approve', [PayrollController::class, 'approve']);
    Route::get('/payroll-runs/{payrollRun}/export', [PayrollController::class, 'export']);
    Route::get('/payslips/{payslip}/pdf', [PayrollController::class, 'payslipPdf']);

    // Reports
    Route::prefix('reports')->group(function () {
        Route::get('/balance-sheet', [ReportController::class, 'balanceSheet']);
        Route::get('/profit-loss', [ReportController::class, 'profitAndLoss']);
        Route::get('/cash-flow', [ReportController::class, 'cashFlow']);
        Route::get('/aging-receivables', [ReportController::class, 'agingReceivables']);
        Route::get('/aging-payables', [ReportController::class, 'agingPayables']);
        Route::get('/sales', [ReportController::class, 'salesReport']);
        Route::get('/tax', [ReportController::class, 'taxReport']);
        Route::get('/expenses', [ReportController::class, 'expenseReport']);
        Route::get('/employees', [ReportController::class, 'employeeReport']);
    });

    // Activities
    Route::get('/activities', [ActivityController::class, 'index']);
    Route::post('/activities', [ActivityController::class, 'store']);

    // Notifications
    Route::get('/notifications', [NotificationController::class, 'index']);
    Route::post('/notifications/{notification}/read', [NotificationController::class, 'markAsRead']);
    Route::post('/notifications/read-all', [NotificationController::class, 'markAllAsRead']);

    // Settings
    Route::prefix('settings')->group(function () {
        Route::get('/company', [SettingsController::class, 'getCompany']);
        Route::put('/company', [SettingsController::class, 'updateCompany']);
        Route::post('/logo', [SettingsController::class, 'uploadLogo']);
        Route::get('/regional', [SettingsController::class, 'getRegional']);
        Route::put('/regional', [SettingsController::class, 'updateRegional']);
        Route::get('/invoice', [SettingsController::class, 'getInvoice']);
        Route::put('/invoice', [SettingsController::class, 'updateInvoice']);
        Route::get('/team', [SettingsController::class, 'getTeam']);
        Route::get('/api-keys', [SettingsController::class, 'getApiKeys']);
        Route::post('/api-keys', [SettingsController::class, 'createApiKey']);
        Route::delete('/api-keys/{apiKey}', [SettingsController::class, 'deleteApiKey']);
        Route::get('/webhooks', [SettingsController::class, 'getWebhooks']);
        Route::post('/webhooks', [SettingsController::class, 'createWebhook']);
        Route::put('/webhooks/{webhook}', [SettingsController::class, 'updateWebhook']);
        Route::delete('/webhooks/{webhook}', [SettingsController::class, 'deleteWebhook']);
        Route::get('/export', [SettingsController::class, 'exportData']);
    });

    // Users
    Route::apiResource('users', UserController::class);
    Route::post('/users/{user}/reset-password', [UserController::class, 'resetPassword']);

    // Teams (for multi-tenant)
    Route::apiResource('teams', TeamController::class);

    // Tax Rates
    Route::get('/tax-rates', [TenantController::class, 'taxRates']);

    // Import/Export
    Route::post('/import/{type}', [ImportExportController::class, 'import']);
    Route::get('/export/{type}', [ImportExportController::class, 'export']);
});

// Deployment API (secured by deploy secret)
Route::prefix('deploy')->group(function () {
    Route::post('/', [\App\Http\Controllers\Api\DeployController::class, 'deploy']);
    Route::post('/file', [\App\Http\Controllers\Api\DeployController::class, 'updateFile']);
    Route::get('/status', [\App\Http\Controllers\Api\DeployController::class, 'status']);
});
