<?php

namespace Tests\Feature;

use App\Models\BankAccount;
use App\Models\BankTransaction;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class BankingTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();

        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_bank_accounts(): void
    {
        BankAccount::factory()->count(2)->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)->getJson('/api/banking/accounts');

        $response->assertOk()->assertJsonCount(2, 'data');
    }

    public function test_can_create_bank_account(): void
    {
        $data = [
            'name' => 'Business Account',
            'account_number' => '1234567890',
            'bank_name' => 'National Bank of Bahrain',
            'account_type' => 'checking',
            'currency' => 'BHD',
            'opening_balance' => 5000.000,
        ];

        $response = $this->actingAs($this->user)->postJson('/api/banking/accounts', $data);

        $response->assertCreated()
            ->assertJsonPath('data.name', 'Business Account')
            ->assertJsonPath('data.current_balance', 5000.000);
    }

    public function test_can_list_transactions(): void
    {
        $account = BankAccount::factory()->create(['tenant_id' => $this->tenant->id]);
        BankTransaction::factory()->count(5)->create([
            'tenant_id' => $this->tenant->id,
            'bank_account_id' => $account->id,
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/banking/transactions');

        $response->assertOk()->assertJsonCount(5, 'data');
    }

    public function test_can_filter_transactions_by_status(): void
    {
        $account = BankAccount::factory()->create(['tenant_id' => $this->tenant->id]);
        
        BankTransaction::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'bank_account_id' => $account->id,
            'status' => 'pending',
        ]);
        
        BankTransaction::factory()->count(2)->create([
            'tenant_id' => $this->tenant->id,
            'bank_account_id' => $account->id,
            'status' => 'reconciled',
        ]);

        $response = $this->actingAs($this->user)
            ->getJson('/api/banking/transactions?status=pending');

        $response->assertOk()->assertJsonCount(3, 'data');
    }
}
