<?php

namespace Tests\Feature;

use App\Models\Bill;
use App\Models\Tenant;
use App\Models\User;
use App\Models\Vendor;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class BillApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Vendor $vendor;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->vendor = Vendor::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_bills(): void
    {
        Bill::factory()->count(5)->create([
            'tenant_id' => $this->tenant->id,
            'vendor_id' => $this->vendor->id,
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/bills');

        $response->assertOk()
            ->assertJsonCount(5, 'data');
    }

    public function test_can_create_bill(): void
    {
        $data = [
            'vendor_id' => $this->vendor->id,
            'bill_number' => 'BILL-001',
            'bill_date' => now()->toDateString(),
            'due_date' => now()->addDays(30)->toDateString(),
            'items' => [
                ['description' => 'Office Supplies', 'quantity' => 1, 'unit_price' => 150.000],
            ],
        ];

        $response = $this->actingAs($this->user)->postJson('/api/bills', $data);

        $response->assertCreated()
            ->assertJsonPath('data.bill_number', 'BILL-001')
            ->assertJsonPath('data.total', 150.000);
    }

    public function test_can_record_bill_payment(): void
    {
        $bill = Bill::factory()->create([
            'tenant_id' => $this->tenant->id,
            'vendor_id' => $this->vendor->id,
            'total' => 500.000,
            'status' => 'received',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/bills/{$bill->id}/payment", [
                'amount' => 500.000,
                'payment_date' => now()->toDateString(),
                'payment_method' => 'bank_transfer',
            ]);

        $response->assertOk();
        $this->assertEquals('paid', $bill->fresh()->status);
    }

    public function test_filters_bills_by_vendor(): void
    {
        $otherVendor = Vendor::factory()->create(['tenant_id' => $this->tenant->id]);
        
        Bill::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'vendor_id' => $this->vendor->id,
        ]);
        Bill::factory()->count(2)->create([
            'tenant_id' => $this->tenant->id,
            'vendor_id' => $otherVendor->id,
        ]);

        $response = $this->actingAs($this->user)
            ->getJson("/api/bills?vendor_id={$this->vendor->id}");

        $response->assertOk()
            ->assertJsonCount(3, 'data');
    }
}
