<?php

namespace Tests\Feature;

use App\Models\Campaign;
use App\Models\Contact;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class CampaignTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();

        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_campaigns(): void
    {
        Campaign::factory()->count(3)->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)->getJson('/api/campaigns');

        $response->assertOk()->assertJsonCount(3, 'data');
    }

    public function test_can_create_email_campaign(): void
    {
        $data = [
            'name' => 'Newsletter Campaign',
            'type' => 'email',
            'subject' => 'Monthly Newsletter',
            'content' => '<h1>Hello {{first_name}}</h1><p>Welcome to our newsletter!</p>',
            'from_name' => 'Setup in Bahrain',
            'from_email' => 'info@setupinbahrain.com',
        ];

        $response = $this->actingAs($this->user)->postJson('/api/campaigns', $data);

        $response->assertCreated()
            ->assertJsonPath('data.name', 'Newsletter Campaign')
            ->assertJsonPath('data.status', 'draft');
    }

    public function test_can_create_sms_campaign(): void
    {
        $data = [
            'name' => 'SMS Promo',
            'type' => 'sms',
            'content' => 'Hi {{first_name}}, check out our latest offers!',
        ];

        $response = $this->actingAs($this->user)->postJson('/api/campaigns', $data);

        $response->assertCreated()
            ->assertJsonPath('data.type', 'sms');
    }

    public function test_cannot_send_non_draft_campaign(): void
    {
        $campaign = Campaign::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'sent',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/campaigns/{$campaign->id}/send");

        $response->assertStatus(422);
    }

    public function test_can_duplicate_campaign(): void
    {
        $campaign = Campaign::factory()->create([
            'tenant_id' => $this->tenant->id,
            'name' => 'Original Campaign',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/campaigns/{$campaign->id}/duplicate");

        $response->assertCreated();
        $this->assertStringContains('Copy', $response->json('data.name'));
    }
}
