<?php

namespace Tests\Feature;

use App\Models\Department;
use App\Models\Employee;
use App\Models\EmployeeSalary;
use App\Models\LeaveRequest;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class EmployeeApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_employees(): void
    {
        Employee::factory()->count(5)->create(['tenant_id' => $this->tenant->id]);

        $response = $this->actingAs($this->user)->getJson('/api/employees');

        $response->assertOk()
            ->assertJsonCount(5, 'data');
    }

    public function test_can_create_employee(): void
    {
        $department = Department::factory()->create(['tenant_id' => $this->tenant->id]);
        
        $data = [
            'first_name' => 'Ahmed',
            'last_name' => 'Al-Khalifa',
            'email' => 'ahmed@example.com',
            'job_title' => 'Software Engineer',
            'department_id' => $department->id,
            'hire_date' => now()->toDateString(),
            'basic_salary' => 1500.000,
        ];

        $response = $this->actingAs($this->user)->postJson('/api/employees', $data);

        $response->assertCreated()
            ->assertJsonPath('data.user.first_name', 'Ahmed');
    }

    public function test_can_filter_by_department(): void
    {
        $dept1 = Department::factory()->create(['tenant_id' => $this->tenant->id]);
        $dept2 = Department::factory()->create(['tenant_id' => $this->tenant->id]);
        
        Employee::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'department_id' => $dept1->id,
        ]);
        Employee::factory()->count(2)->create([
            'tenant_id' => $this->tenant->id,
            'department_id' => $dept2->id,
        ]);

        $response = $this->actingAs($this->user)
            ->getJson("/api/employees?department_id={$dept1->id}");

        $response->assertOk()
            ->assertJsonCount(3, 'data');
    }

    public function test_can_update_salary(): void
    {
        $employee = Employee::factory()->create(['tenant_id' => $this->tenant->id]);
        EmployeeSalary::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $employee->id,
            'basic_salary' => 1000,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/employees/{$employee->id}/salary", [
                'basic_salary' => 1500.000,
                'effective_date' => now()->toDateString(),
            ]);

        $response->assertOk();
        $this->assertEquals(1500.000, $employee->currentSalary()->basic_salary);
    }

    public function test_can_terminate_employee(): void
    {
        $employee = Employee::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'active',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/employees/{$employee->id}/terminate", [
                'termination_date' => now()->toDateString(),
                'reason' => 'Resignation',
            ]);

        $response->assertOk();
        $this->assertEquals('terminated', $employee->fresh()->status);
    }
}
