<?php

namespace Tests\Feature;

use App\Models\Contact;
use App\Models\Invoice;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class InvoiceApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Contact $contact;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_invoices(): void
    {
        Invoice::factory()->count(5)->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/invoices');

        $response->assertOk()
            ->assertJsonCount(5, 'data');
    }

    public function test_can_create_invoice(): void
    {
        $data = [
            'contact_id' => $this->contact->id,
            'invoice_date' => now()->toDateString(),
            'due_date' => now()->addDays(30)->toDateString(),
            'items' => [
                [
                    'description' => 'Company Formation Services',
                    'quantity' => 1,
                    'unit_price' => 950.000,
                ],
            ],
        ];

        $response = $this->actingAs($this->user)->postJson('/api/invoices', $data);

        $response->assertCreated()
            ->assertJsonPath('data.status', 'draft')
            ->assertJsonPath('data.total', 950.000);
    }

    public function test_can_send_invoice(): void
    {
        $invoice = Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'status' => 'draft',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/invoices/{$invoice->id}/send");

        $response->assertOk();
        $this->assertEquals('sent', $invoice->fresh()->status);
    }

    public function test_can_record_payment(): void
    {
        $invoice = Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'status' => 'sent',
            'total' => 1000.000,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/invoices/{$invoice->id}/payment", [
                'amount' => 500.000,
                'payment_date' => now()->toDateString(),
                'payment_method' => 'bank_transfer',
            ]);

        $response->assertOk();
        $this->assertEquals('partial', $invoice->fresh()->status);
        $this->assertEquals(500.000, $invoice->fresh()->amount_paid);
    }

    public function test_full_payment_marks_invoice_paid(): void
    {
        $invoice = Invoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'status' => 'sent',
            'total' => 1000.000,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/invoices/{$invoice->id}/payment", [
                'amount' => 1000.000,
                'payment_date' => now()->toDateString(),
                'payment_method' => 'bank_transfer',
            ]);

        $response->assertOk();
        $this->assertEquals('paid', $invoice->fresh()->status);
    }

    public function test_cannot_access_other_tenant_invoices(): void
    {
        $otherTenant = Tenant::factory()->create();
        $invoice = Invoice::factory()->create([
            'tenant_id' => $otherTenant->id,
            'contact_id' => Contact::factory()->create(['tenant_id' => $otherTenant->id])->id,
        ]);

        $response = $this->actingAs($this->user)
            ->getJson("/api/invoices/{$invoice->id}");

        $response->assertForbidden();
    }
}
