<?php

namespace Tests\Feature;

use App\Models\Milestone;
use App\Models\Project;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class MilestoneApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Project $project;

    protected function setUp(): void
    {
        parent::setUp();
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->project = Project::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_list_milestones(): void
    {
        Milestone::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/milestones');

        $response->assertOk()->assertJsonCount(3, 'data');
    }

    public function test_can_create_milestone(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/milestones', [
            'project_id' => $this->project->id,
            'name' => 'Phase 1 Complete',
            'due_date' => now()->addMonth()->toDateString(),
        ]);

        $response->assertCreated()
            ->assertJsonPath('data.name', 'Phase 1 Complete');
    }

    public function test_can_complete_milestone(): void
    {
        $milestone = Milestone::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $this->project->id,
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/milestones/{$milestone->id}/complete");

        $response->assertOk();
        $this->assertNotNull($milestone->fresh()->completed_at);
    }
}
