<?php

namespace Tests\Feature;

use App\Models\Employee;
use App\Models\EmployeeSalary;
use App\Models\PayrollRun;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class PayrollTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;

    protected function setUp(): void
    {
        parent::setUp();

        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_create_payroll_run(): void
    {
        $data = [
            'pay_period_start' => now()->startOfMonth()->toDateString(),
            'pay_period_end' => now()->endOfMonth()->toDateString(),
            'payment_date' => now()->endOfMonth()->addDays(5)->toDateString(),
        ];

        $response = $this->actingAs($this->user)->postJson('/api/payroll/runs', $data);

        $response->assertCreated()
            ->assertJsonPath('data.status', 'draft');
    }

    public function test_can_process_payroll_run(): void
    {
        $employee = Employee::factory()->create([
            'tenant_id' => $this->tenant->id,
            'user_id' => User::factory()->create(['tenant_id' => $this->tenant->id])->id,
            'nationality' => 'BH',
        ]);

        EmployeeSalary::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $employee->id,
            'basic_salary' => 1000.000,
            'housing_allowance' => 200.000,
        ]);

        $run = PayrollRun::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'draft',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/payroll/runs/{$run->id}/process");

        $response->assertOk();
        $this->assertEquals('processed', $run->fresh()->status);
        $this->assertEquals(1, $run->items()->count());
    }

    public function test_gosi_calculated_correctly_for_bahraini(): void
    {
        $employee = Employee::factory()->create([
            'tenant_id' => $this->tenant->id,
            'user_id' => User::factory()->create(['tenant_id' => $this->tenant->id])->id,
            'nationality' => 'BH',
        ]);

        EmployeeSalary::factory()->create([
            'tenant_id' => $this->tenant->id,
            'employee_id' => $employee->id,
            'basic_salary' => 1000.000,
        ]);

        $run = PayrollRun::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'draft',
        ]);

        $this->actingAs($this->user)->postJson("/api/payroll/runs/{$run->id}/process");

        $item = $run->items()->first();
        $this->assertEquals(70.000, $item->gosi_employee); // 7%
        $this->assertEquals(120.000, $item->gosi_employer); // 12%
    }

    public function test_can_approve_payroll_run(): void
    {
        $run = PayrollRun::factory()->create([
            'tenant_id' => $this->tenant->id,
            'status' => 'processed',
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/payroll/runs/{$run->id}/approve");

        $response->assertOk();
        $this->assertEquals('approved', $run->fresh()->status);
    }
}
