<?php

namespace Tests\Feature;

use App\Models\Contact;
use App\Models\RecurringInvoice;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class RecurringInvoiceApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Contact $contact;

    protected function setUp(): void
    {
        parent::setUp();
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $this->contact = Contact::factory()->create(['tenant_id' => $this->tenant->id]);
    }

    public function test_can_create_recurring_invoice(): void
    {
        $response = $this->actingAs($this->user)->postJson('/api/recurring-invoices', [
            'contact_id' => $this->contact->id,
            'frequency' => 'monthly',
            'start_date' => now()->toDateString(),
            'items' => [
                ['description' => 'Monthly Service', 'quantity' => 1, 'unit_price' => 500.000],
            ],
        ]);

        $response->assertCreated()
            ->assertJsonPath('data.frequency', 'monthly');
    }

    public function test_can_pause_recurring_invoice(): void
    {
        $recurring = RecurringInvoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'is_active' => true,
        ]);

        $response = $this->actingAs($this->user)
            ->putJson("/api/recurring-invoices/{$recurring->id}", ['is_active' => false]);

        $response->assertOk();
        $this->assertFalse($recurring->fresh()->is_active);
    }

    public function test_can_generate_invoice_from_recurring(): void
    {
        $recurring = RecurringInvoice::factory()->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'is_active' => true,
            'next_invoice_date' => now()->toDateString(),
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/recurring-invoices/{$recurring->id}/generate");

        $response->assertOk()
            ->assertJsonStructure(['data' => ['id', 'invoice_number']]);
    }

    public function test_can_get_recurring_invoice_stats(): void
    {
        RecurringInvoice::factory()->count(3)->create([
            'tenant_id' => $this->tenant->id,
            'contact_id' => $this->contact->id,
            'is_active' => true,
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/recurring-invoices/stats');

        $response->assertOk()
            ->assertJsonStructure(['data' => ['active', 'monthly_value']]);
    }
}
