<?php

namespace Tests\Feature;

use App\Models\Project;
use App\Models\Task;
use App\Models\TaskStatus;
use App\Models\Tenant;
use App\Models\TimeEntry;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class TimeEntryApiTest extends TestCase
{
    use RefreshDatabase;

    protected User $user;
    protected Tenant $tenant;
    protected Task $task;

    protected function setUp(): void
    {
        parent::setUp();
        
        $this->tenant = Tenant::factory()->create();
        $this->user = User::factory()->create(['tenant_id' => $this->tenant->id]);
        $project = Project::factory()->create(['tenant_id' => $this->tenant->id]);
        $status = TaskStatus::factory()->create(['project_id' => $project->id]);
        $this->task = Task::factory()->create([
            'tenant_id' => $this->tenant->id,
            'project_id' => $project->id,
            'status_id' => $status->id,
        ]);
    }

    public function test_can_start_timer(): void
    {
        $response = $this->actingAs($this->user)
            ->postJson('/api/time-entries/start', [
                'task_id' => $this->task->id,
                'description' => 'Working on feature',
            ]);

        $response->assertCreated()
            ->assertJsonPath('data.is_running', true)
            ->assertJsonPath('data.task_id', $this->task->id);
    }

    public function test_can_stop_timer(): void
    {
        $entry = TimeEntry::factory()->create([
            'tenant_id' => $this->tenant->id,
            'user_id' => $this->user->id,
            'task_id' => $this->task->id,
            'is_running' => true,
            'start_time' => now()->subHour(),
        ]);

        $response = $this->actingAs($this->user)
            ->postJson("/api/time-entries/{$entry->id}/stop");

        $response->assertOk()
            ->assertJsonPath('data.is_running', false);
        
        $this->assertNotNull($entry->fresh()->end_time);
        $this->assertGreaterThan(0, $entry->fresh()->duration);
    }

    public function test_can_create_manual_entry(): void
    {
        $response = $this->actingAs($this->user)
            ->postJson('/api/time-entries', [
                'task_id' => $this->task->id,
                'start_time' => now()->subHours(2)->toDateTimeString(),
                'end_time' => now()->subHour()->toDateTimeString(),
                'description' => 'Code review',
            ]);

        $response->assertCreated()
            ->assertJsonPath('data.duration', 60);
    }

    public function test_can_get_running_timer(): void
    {
        TimeEntry::factory()->create([
            'tenant_id' => $this->tenant->id,
            'user_id' => $this->user->id,
            'task_id' => $this->task->id,
            'is_running' => true,
        ]);

        $response = $this->actingAs($this->user)->getJson('/api/time-entries/running');

        $response->assertOk()
            ->assertJsonPath('data.is_running', true);
    }

    public function test_can_get_user_time_summary(): void
    {
        TimeEntry::factory()->count(5)->create([
            'tenant_id' => $this->tenant->id,
            'user_id' => $this->user->id,
            'task_id' => $this->task->id,
            'duration' => 60,
            'is_running' => false,
        ]);

        $response = $this->actingAs($this->user)
            ->getJson('/api/time-entries/summary?start_date=' . now()->startOfMonth()->toDateString());

        $response->assertOk()
            ->assertJsonPath('data.total_hours', 5.0);
    }
}
